// client/src/components/VoiceAssistantOverlay.jsx
// Bottom-sheet style voice overlay (covers ~2/3 screen), elegant and elder-friendly

import React, { useEffect } from "react";

/**
 * Props:
 * - open: boolean - show/hide overlay
 * - title?: string - overlay headline
 * - transcript?: string[] - partial/streaming STT results
 * - status?: 'idle' | 'listening' | 'thinking' | 'error'
 * - onClose?: () => void
 * - onCallCaregiver?: () => void
 * - onImOk?: () => void
 */
export default function VoiceAssistantOverlay({
  open,
  title = "What's wrong?",
  transcript = [],
  status = "listening",
  onClose,
  onCallCaregiver,
  onImOk,
}) {
  useEffect(() => {
    function onKey(e) {
      if (e.key === "Escape" && onClose) onClose();
    }
    if (open) window.addEventListener("keydown", onKey);
    return () => window.removeEventListener("keydown", onKey);
  }, [open, onClose]);

  if (!open) return null;

  return (
    <div role="dialog" aria-modal="true" aria-label="DORA Voice Assistant" style={styles.root}>
      <div style={styles.backdrop} onClick={onClose} aria-hidden />
      <section style={styles.sheet} aria-label="Voice assistant sheet">
        <div style={styles.handleWrap} aria-hidden>
          <div style={styles.handle} />
        </div>

        <div style={styles.headerRow}>
          <div style={styles.pulseWrap} aria-hidden>
            <div
              style={{
                ...styles.pulseDot,
                ...(status === "listening" ? styles.pulseAnim : {}),
              }}
            />
          </div>
          <h1 style={styles.title}>
            {title} <span style={styles.titleSub}>(Please tell me what's wrong)</span>
          </h1>
        </div>

        <div style={styles.transcriptBox} aria-live="polite">
          {transcript.length === 0 ? (
            <p style={styles.hintText}>
              I'm listening... You can say: "Call caregiver / I'm OK / Turn on living room light"
            </p>
          ) : (
            <ul style={styles.transcriptList}>
              {transcript.slice(-5).map((t, i) => (
                <li key={i} style={styles.transcriptItem}>
                  {t}
                </li>
              ))}
            </ul>
          )}
        </div>

        <div style={styles.actionsRow}>
          <button
            type="button"
            onClick={onCallCaregiver}
            style={styles.primaryAction}
            aria-label="Call caregiver now"
          >
            📞 CALL CAREGIVER
          </button>
          <button
            type="button"
            onClick={onImOk}
            style={styles.secondaryAction}
            aria-label="I'm OK"
          >
            ✅ I'M OK
          </button>
          <button
            type="button"
            onClick={onClose}
            style={styles.ghostAction}
            aria-label="Close"
          >
            Cancel
          </button>
        </div>

        <div style={styles.footerRow}>
          <span style={styles.statusPill}>
            {status === "listening" && "🎤 Listening"}
            {status === "thinking" && "🤔 Understanding"}
            {status === "error" && "⚠️ Voice unclear, please speak again"}
            {status === "idle" && "🟢 Ready"}
          </span>
          <span style={styles.a11yNote}>Press Esc to close</span>
        </div>
      </section>
    </div>
  );
}

const styles = {
  root: {
    position: "fixed",
    inset: 0,
    zIndex: 9999,
  },
  // Backdrop gradient: darker at bottom, more transparent at top (keeps top 1/3 visible)
  backdrop: {
    position: "absolute",
    inset: 0,
    background:
      "linear-gradient(to top, rgba(0,0,0,0.50) 0%, rgba(0,0,0,0.19) 35%, rgba(0,0,0,0.06) 55%, rgba(0,0,0,0) 100%)",
    backdropFilter: "blur(2px)",
    WebkitBackdropFilter: "blur(2px)",
  },
  // Bottom sheet: anchored to bottom, height ≈ 66vh
  sheet: {
    position: "absolute",
    left: "50%",
    transform: "translateX(-50%)",
    bottom: 0,
    width: "min(1040px, 96vw)",
    height: "66vh",
    borderTopLeftRadius: 28,
    borderTopRightRadius: 28,
    background:
      "linear-gradient(180deg, #1E293B 0%, #0F172A 100%) padding-box, linear-gradient(180deg, rgba(59,130,246,0.4), rgba(244,63,94,0.4)) border-box",
    border: "2px solid transparent",
    boxShadow:
      "0 -20px 60px rgba(0,0,0,0.5), inset 0 1px 0 rgba(255,255,255,0.08), inset 0 -1px 0 rgba(0,0,0,0.3)",
    color: "#E5E7EB",
    padding: "20px 24px 18px",
    display: "flex",
    flexDirection: "column",
    overflow: "hidden",
  },
  handleWrap: {
    display: "grid",
    placeItems: "center",
    marginBottom: 4,
  },
  handle: {
    width: 64,
    height: 6,
    borderRadius: 999,
    background: "rgba(148,163,184,0.6)",
    marginBottom: 4,
  },
  headerRow: {
    display: "flex",
    alignItems: "center",
    gap: 16,
    paddingTop: 8,
    marginBottom: 4,
  },
  pulseWrap: {
    width: 42,
    height: 42,
    position: "relative",
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
  },
  pulseDot: {
    width: 14,
    height: 14,
    borderRadius: 999,
    background: "#22C55E",
    position: "absolute",
    boxShadow: "0 0 0 0 rgba(34,197,94,0.6)",
  },
  pulseAnim: {
    animation: "va-pulse 1.8s ease-out infinite",
  },
  title: {
    fontSize: 32,
    margin: 0,
    lineHeight: 1.2,
    letterSpacing: 0.1,
    color: "#F8FAFC",
    textShadow: "0 1px 2px rgba(0,0,0,0.5)",
    flex: 1,
  },
  titleSub: {
    fontSize: 16,
    color: "#94A3B8",
    marginLeft: 8,
    fontWeight: 400,
  },
  transcriptBox: {
    marginTop: 16,
    background:
      "linear-gradient(180deg, rgba(30,41,59,0.8), rgba(15,23,42,0.6))",
    border: "1px solid rgba(148,163,184,0.2)",
    borderRadius: 16,
    minHeight: 120,
    maxHeight: 200,
    padding: "14px 16px",
    overflow: "auto",
    flex: 1,
  },
  hintText: {
    margin: 0,
    color: "#A3B2C7",
    fontSize: 18,
    lineHeight: 1.5,
  },
  transcriptList: {
    margin: 0,
    padding: 0,
    listStyle: "none",
    display: "flex",
    gap: 8,
    flexWrap: "wrap",
  },
  transcriptItem: {
    fontSize: 20,
    fontWeight: "bold",
    padding: "12px 16px",
    borderRadius: 16,
    border: "2px solid rgba(96,165,250,0.6)",
    background: "linear-gradient(180deg, #3B82F6, #2563EB)",
    color: "#FFFFFF",
    boxShadow: "0 4px 12px rgba(59,130,246,0.4), 0 2px 4px rgba(0,0,0,0.3)",
    textShadow: "0 1px 2px rgba(0,0,0,0.3)",
  },
  actionsRow: {
    marginTop: 16,
    display: "grid",
    gridTemplateColumns: "1fr 1fr auto",
    gap: 12,
    flexShrink: 0,
  },
  primaryAction: {
    fontSize: 20,
    fontWeight: 800,
    padding: "16px 18px",
    borderRadius: 16,
    border: "1px solid rgba(248,113,113,0.5)",
    background: "linear-gradient(180deg, #EF4444 0%, #DC2626 100%)",
    color: "#FFF",
    cursor: "pointer",
    boxShadow:
      "0 8px 20px rgba(239,68,68,0.4), inset 0 1px 0 rgba(255,255,255,0.2)",
    transition: "all 0.2s ease",
  },
  secondaryAction: {
    fontSize: 20,
    fontWeight: 700,
    padding: "16px 18px",
    borderRadius: 16,
    border: "1px solid rgba(34,197,94,0.4)",
    background:
      "linear-gradient(180deg, rgba(34,197,94,0.25), rgba(22,163,74,0.25))",
    color: "#D1FAE5",
    cursor: "pointer",
    transition: "all 0.2s ease",
  },
  ghostAction: {
    fontSize: 18,
    fontWeight: 600,
    padding: "16px 18px",
    borderRadius: 12,
    border: "1px solid rgba(148,163,184,0.4)",
    background: "transparent",
    color: "#B6C2D6",
    cursor: "pointer",
    transition: "all 0.2s ease",
  },
  footerRow: {
    marginTop: 12,
    display: "flex",
    justifyContent: "space-between",
    alignItems: "center",
    flexShrink: 0,
  },
  statusPill: {
    fontSize: 16,
    padding: "8px 12px",
    border: "1px solid rgba(148,163,184,0.35)",
    borderRadius: 999,
    color: "#C9D5E9",
    background: "rgba(15,23,42,0.6)",
    boxShadow: "0 2px 4px rgba(0,0,0,0.2)",
  },
  a11yNote: {
    fontSize: 14,
    color: "#94A3B8",
  },
};

// Inject keyframes once
const styleElId = "va-pulse-keyframes";
if (typeof document !== "undefined" && !document.getElementById(styleElId)) {
  const style = document.createElement("style");
  style.id = styleElId;
  style.innerHTML = `
@keyframes va-pulse {
  0% { box-shadow: 0 0 0 0 rgba(34,197,94,0.6); }
  70% { box-shadow: 0 0 0 22px rgba(34,197,94,0); }
  100% { box-shadow: 0 0 0 0 rgba(34,197,94,0); }
}`;
  document.head.appendChild(style);
}
