clc; 
clear; 
close all;

%% === DATI DI INPUT ===

% Salita
mach_climb = [0.35 0.7 1 1.5 2 2.5 3 3.5 4 5 6 6]; % Mach
alt_climb = [0 2600 6400 10500 12850 15700 18000 19900 21600 24500 27000 28600]; % in m

% Discesa
mach_desc = [4 3.5 3 2.5 2 1.5 1 0.8 0.5 0.4]; % Mach
alt_desc = [28500 28000 26700 24200 21300 17700 12700 9800 3800 460]; % in m

% Distanze [km]
dist_climb = 2069;
dist_cruise = 6790;
dist_desc = 1141;

% Costanti
gamma = 1.4;
R = 287;

% isa_temp = @(h) max(288.15 - 0.0065 * h, 216.65);
mach2vel = @(M, T) M .* sqrt(gamma * R .* T);

%% === CALCOLO VELOCITÀ ===

T_climb = isa_temp(alt_climb);
T_desc = isa_temp(alt_desc);

v_climb = mach2vel(mach_climb, T_climb);
v_desc = mach2vel(mach_desc, T_desc);

%% === DISTANZE E TEMPI ===

% -- Salita
n_climb = length(alt_climb);
dx_climb = ones(1, n_climb-1);
dx_climb = dx_climb.* dist_climb * 1000 / sum(dx_climb);
v_avg_climb = (v_climb(1:end-1) + v_climb(2:end)) / 2;
dt_climb = dx_climb ./ v_avg_climb;

% -- Crociera
alt_cruise = 28600; % m
mach_cruise = 6;
T_cruise = isa_temp(alt_cruise);
v_cruise = mach2vel(mach_cruise, T_cruise);
dt_cruise = (dist_cruise * 1000) / v_cruise;

% -- Discesa
n_desc = length(alt_desc);
dx_desc = ones(1, n_desc-1);
dx_desc = dx_desc.* dist_desc * 1000 / sum(dx_desc);
v_avg_desc = (v_desc(1:end-1) + v_desc(2:end)) / 2;
dt_desc = dx_desc ./ v_avg_desc;

%% === CREAZIONE VETTORI ===

% Tempo
t_climb = [0, cumsum(dt_climb)] / 60;
t_cruise = linspace(t_climb(end), t_climb(end) + dt_cruise/60, 10);
t_desc = t_cruise(end) + [0, cumsum(dt_desc)] / 60;

% Altitudine
alt_cruise_vec = repmat(alt_cruise, 1, length(t_cruise));
alt_total = [alt_climb, alt_cruise_vec, alt_desc];

% Distanza
x_climb = [0, cumsum(dx_climb)] / 1000;
x_cruise = linspace(x_climb(end), x_climb(end) + dist_cruise, length(t_cruise));
x_desc = x_cruise(end) + [0, cumsum(dx_desc)] / 1000;

% Unifica per plotting
x_total = [x_climb, x_cruise, x_desc];
t_total = [t_climb, t_cruise, t_desc];

% Verifica dimensioni
assert(length(x_total) == length(alt_total), 'Errore: x_total e alt_total hanno lunghezze diverse');
assert(length(t_total) == length(alt_total), 'Errore: t_total e alt_total hanno lunghezze diverse');


%% === GRAFICI ===

% 1 Distanza vs Altitudine

figure;

plot(x_total, alt_total / 1000, 'g', 'LineWidth', 2);
xlabel('Distance [km]');
ylabel('Altitude [km]');
title('Altitude vs Distance plot');
grid on;

% 2 Tempo vs Altitudine
figure;

plot(t_total, alt_total / 1000, 'c', 'LineWidth', 2);
xlabel('Time [min]');
ylabel('Altitude [km]');
title('Altitude vs Time plot');
grid on;

% 3 Mach vs Altitudine
figure;

plot(mach_climb, alt_climb / 1000, 'b-o', 'LineWidth', 2, 'DisplayName', 'Climb');
hold on;
plot([6, mach_desc(1)], [28600, alt_desc(1)] / 1000, 'm--*', 'LineWidth', 2, 'DisplayName', 'Initial descent');
plot(mach_desc, alt_desc / 1000, 'r-s', 'LineWidth', 2, 'DisplayName', 'Descent');
xlabel('Mach');
ylabel('Altitude [km]');
title('Mach vs Altitude plot');
legend('Location','northwest');
grid on;


%% === Funzioni ===

function T = isa_temp(h)
    if h < 11000
        T = 288.15 - 0.0065 * h;
    elseif h < 20000
        T = 216.65;
    elseif h < 32000
        T = 216.65 + 0.001 * (h - 20000);
    elseif h < 47000
        T = 228.65 + 0.0028 * (h - 32000);
    else
        T = 270;
    end
end