
#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Classificazione HRV con SVM (RBF) + RobustScaler
Fatica: riposo / media_fatica / alta_fatica
Validazione Leave-One-Out (LOOCV) e salvataggio dei risultati in Excel.

Autore: Annarosa Scalcione
Data: 2025-09-20

Funzionalità:
1) Carica il dataset HRV (features, etichette, soggetti, RPE)
2) Applica una pipeline: imputazione (mediana) + scaling robusto + SVM (RBF)
3) Valida tramite Leave-One-Out (una iterazione per soggetto)
4) Calcola accuratezza bilanciata, report e matrice di confusione
5) Esporta i risultati in file Excel con 4 fogli:
   - Report di classificazione
   - Matrice di confusione
   - Predizioni singole
   - Balanced Accuracy complessiva
"""

# ============================================================
# LIBRERIE
# ============================================================
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns

from sklearn.pipeline import Pipeline
from sklearn.impute import SimpleImputer
from sklearn.preprocessing import RobustScaler
from sklearn.svm import SVC
from sklearn.metrics import classification_report, balanced_accuracy_score, confusion_matrix
from sklearn.model_selection import LeaveOneOut


# ============================================================
# FUNZIONE PRINCIPALE DI VALUTAZIONE
# ============================================================

def svm_robust_evaluation(df, features, output_path="svm_robust_results.xlsx"):
    """
    Esegue classificazione HRV con SVM (RBF) + RobustScaler in schema LOOCV.

    Parametri
    ---------
    df : pandas.DataFrame
        Dataset contenente le feature HRV e colonne 'label', 'subject_id', 'RPE'
    features : list of str
        Lista delle feature numeriche da usare per l'addestramento
    output_path : str
        Percorso del file Excel dove salvare i risultati
    """
    X = df[features].astype(float).values
    y = df["label"].astype(str).values
    subjects = df["subject_id"].values
    rpe_vals = df["RPE"].values

    loo = LeaveOneOut()
    y_true, y_pred = [], []
    subj_list, rpe_list = [], []
    p_r_list, p_m_list, p_a_list, I_list = [], [], [], []

    # Validazione Leave-One-Out: ogni soggetto usato a turno come test
    for train_idx, test_idx in loo.split(X):
        pipe = Pipeline([
            ("imputer", SimpleImputer(strategy="median")),
            ("scaler", RobustScaler()),
            ("clf", SVC(kernel="rbf", probability=True, class_weight="balanced", random_state=42))
        ])
        pipe.fit(X[train_idx], y[train_idx])
        preds = pipe.predict(X[test_idx])
        proba = pipe.predict_proba(X[test_idx])

        # Estrae probabilità per ciascuna classe (riposo, media, alta fatica)
        classes = list(pipe.classes_)
        i_r = classes.index("riposo")
        i_m = classes.index("media_fatica")
        i_a = classes.index("alta_fatica")

        P_r = proba[0, i_r]
        P_m = proba[0, i_m]
        P_a = proba[0, i_a]
        I = P_a - P_r  # indice di intensità (proxy graduale da riposo ad alta fatica)

        # Accumula i risultati
        y_true.extend(y[test_idx])
        y_pred.extend(preds)
        subj_list.extend(subjects[test_idx])
        rpe_list.extend(rpe_vals[test_idx])
        p_r_list.append(P_r)
        p_m_list.append(P_m)
        p_a_list.append(P_a)
        I_list.append(I)

    # ============================================================
    # METRICHE E RISULTATI
    # ============================================================

    bal_acc = balanced_accuracy_score(y_true, y_pred)
    report_dict = classification_report(y_true, y_pred, output_dict=True)

    # Matrice di confusione
    labels = np.unique(y_true)
    cm = confusion_matrix(y_true, y_pred, labels=labels)
    cm_df = pd.DataFrame(cm, index=labels, columns=labels)

    # Predizioni campione per campione
    preds_df = pd.DataFrame({
        "subject_id": subj_list,
        "True_Label": y_true,
        "Predicted_Label": y_pred,
        "RPE_reale": rpe_list,
        "P_riposo": p_r_list,
        "P_media": p_m_list,
        "P_alta": p_a_list,
        "IntensityIndex": I_list,
        "Correct": np.array(y_true) == np.array(y_pred)
    })

    # ============================================================
    # SALVATAGGIO RISULTATI
    # ============================================================
    with pd.ExcelWriter(output_path) as writer:
        pd.DataFrame(report_dict).transpose().to_excel(writer, sheet_name="Report")
        cm_df.to_excel(writer, sheet_name="Confusion_Matrix")
        preds_df.to_excel(writer, sheet_name="Predizioni", index=False)
        pd.DataFrame({"Balanced Accuracy": [bal_acc]}).to_excel(writer, sheet_name="Balanced_Accuracy", index=False)

    # ============================================================
    # OUTPUT CONSOLE E GRAFICO
    # ============================================================

    print("\n=== SVM (RBF) + RobustScaler ===")
    print("Balanced Accuracy:", round(bal_acc, 3))
    print(classification_report(y_true, y_pred, digits=3))

    # Plot della matrice di confusione
    plt.figure(figsize=(5, 4))
    sns.heatmap(cm, annot=True, fmt="d", cmap="Blues", xticklabels=labels, yticklabels=labels)
    plt.xlabel("Predetto")
    plt.ylabel("Reale")
    plt.title("Confusion Matrix - SVM (RBF) + RobustScaler")
    plt.tight_layout()
    plt.show()

    print(f"\nRisultati salvati in: {output_path}")


# ============================================================
# MAIN SCRIPT
# ============================================================
if __name__ == "__main__":
    # Percorso del file Excel con features HRV e RPE
    df = pd.read_excel(
        "/Users/annarosascalcione/Desktop/università/Magistrale/Tesi Anna/codici/soggetti/"
        "D2_time_domain_non_linear_with_RPE.xlsx"
    )

    # Rinominazione colonne standard
    df.rename(columns={"SOGGETTO": "subject_id", "FASE": "label"}, inplace=True)

    # Selezione delle feature numeriche per il modello
    features = [c for c in df.columns if c not in ["subject_id", "label", "RPE"]]

    # Esecuzione valutazione
    svm_robust_evaluation(
        df,
        features,
        output_path=(
            "/Users/annarosascalcione/Desktop/università/Magistrale/Tesi Anna/codici/soggetti/"
            "svm_robust_results.xlsx"
        )
    )
