

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Interpolazione percentuale VO₂max in funzione della %FCmax
Autore: Annarosa Scalcione
Data: 2025-09-29

Funzionalità:
1) Carica i dati dei soggetti (FCmax, VO₂max)
2) Applica una curva di riferimento %FCmax → %VO₂max basata su valori medi noti
3) Interpola i valori osservati per stimare la %VO₂max corrispondente
4) Calcola il VO₂ assoluto stimato per ciascun soggetto

Utile per:
- Confrontare i livelli di intensità del protocollo con le soglie relative di VO₂
- Validare la suddivisione delle fasce di sforzo (bassa, media, alta intensità)
"""

# ======================================================
# LIBRERIE
# ======================================================
import numpy as np
import pandas as pd

# ======================================================
# DATI DI INPUT
# ======================================================

# Dati sperimentali per ciascun soggetto
data = {
    "Soggetto": [1, 3, 6, 7, 8, 10, 11, 12],
    "%FCmax": [48, 44, 45, 42, 39, 68, 41, 57],
    "VO2max": [51.4, 42.5, 60.5, 39.4, 51.4, 49.9, 29.4, 53.4]
}

df = pd.DataFrame(data)

# ======================================================
# CURVA DI RIFERIMENTO
# ======================================================
# Aggiungo punto iniziale (0% FCmax = 0% VO₂max) per evitare plateau sotto il 50%
fc_ref = np.array([0, 50, 60, 70, 80, 90, 100])    # %FCmax
vo2_ref = np.array([0, 28, 40, 58, 70, 83, 100])   # %VO₂max

# ======================================================
# INTERPOLAZIONE LINEARE
# ======================================================
# Interpola la %VO₂max per ciascun valore osservato di %FCmax
df["%VO2max_stimato"] = np.interp(df["%FCmax"], fc_ref, vo2_ref)

# Calcolo del VO₂ stimato (ml/kg/min)
df["VO2_stimato"] = df["VO2max"] * (df["%VO2max_stimato"] / 100)

# ======================================================
# OUTPUT
# ======================================================
print(df)
