function [Hn, Hn_env] = multiple_filter(Trace, T, freq, std)
% This function builds the frequency-time (envelope) signal by the multiple 
% filter technique (using a series of narrow band-pass Gaussian filter 
% according to Dziewonski et al 1969).
%
% % INPUT:
%        Trace: A sigle trace column vector with nt samples.
%            T: A vector represent the time serie.
%         freq: A vector with nfreq columns specify the frequency range (Hz).
%          std: The standard deviation of the narrow band Gaussian window.
%
% % OUTPUT:
%           Hn: The filtered signal with nt rows and nfreq columns 
%       Hn_env: The evelope matrix of Hn, calculated by Hilbert transform

% Make sure the frequency vector freq is a row vector
freq = freq(:)';

% Useful parameters
nt = length(T);
nfreq = length(freq);

% FFT parameters
nfft = 2 ^ nextpow2(nt);
dt = T(2) - T(1);
fn = .5 / dt;
nf = nfft/2 + 1;
f = fn .* linspace(0, 1, nf)';

% Gaussian Filter
omegas = 2 * pi .* f;
omega0 = 2 * pi .* freq;
filter = exp(-.5 .* ((omegas - omega0) ./ (std .* omega0)).^2);

% FFT transform and filtering
dfft = fft(Trace, nfft, 1);
dfilter = zeros(nfft, nfreq);
dfilter(1:nf,:) = dfft(1:nf) .* filter;                 % Positive part
dfilter(nf+1:end,:) = flipud(conj(dfilter(2:nf-1,:)));  % Negative part

% IFFT
Hn = real(ifft(dfilter));
Hn = Hn(1:nt, :);

% Envelope
if nargout > 1   
    h  = zeros(nfft,1); 
	h([1 nf]) = 1;
	h(2:nf-1) = 2;
    An = abs(ifft(dfilter .* h(:,ones(1,nfreq)),[],1));          % Envelope
    Hn_env = An(1:nt, :);
    Hn_env = Hn_env ./ max(Hn_env);           % Normalize in each frequency
end

end
