%% Mean amplitude spectra over ALL pairs with ±1 std shading
% Needs: CNR_CC_mat.mat  (matrix CC_mat [nt x nPairs])

clear; close all; clc

%% --- Load data & basics ---
load('CNR_CC_mat.mat');                 % CC_mat [nt x nPairs]
[nt, nPairs] = size(CC_mat);

dt  = 0.001;                             % s
fs  = 1/dt;
n   = nt;
nf  = floor(n/2);
f   = (0:nf).' * (fs/n);                 % 0..fs/2 (nf+1 points)

% --- Gaussian multi band-pass params (2–100 Hz) ---
fmin  = 2;      fmax = 100;      df0 = 0.5;
sigma = 0.15;
f0    = (fmin:df0:fmax);                 % row vector

% Composite amplitude response on positive freqs
G    = exp(-0.5 * ((f - f0) ./ (sigma .* f0)).^2);
Wpos = max(G, [], 2);
Wpos(f < fmin) = 0;  Wpos(f > fmax) = 0;

% Symmetric full H(f) for real signal length n
H = zeros(n,1);
H(1:nf+1) = Wpos;
if mod(n,2)==0
    H(nf+2:end) = Wpos(end-1:-1:2);
else
    H(nf+2:end) = Wpos(end:-1:2);
end

%% --- Accumulators for mean/std (before & after) ---
Sraw_sum   = zeros(nf+1,1);
Sraw_sumsq = zeros(nf+1,1);
Sf_sum     = zeros(nf+1,1);
Sf_sumsq   = zeros(nf+1,1);

%% --- Main loop over all pairs ---
for ip = 1:nPairs
    % Base preproc: de-mean + detrend
    x = CC_mat(:,ip);
    x = detrend(x - mean(x));

    % FFT
    X = fft(x);

    % One-sided amplitude spectrum
    Sx = abs(X(1:nf+1));
    if nf+1 > 2
        Sx(2:end-1) = 2*Sx(2:end-1);
    end

    % Apply composite filter
    Xf = X .* H;
    Sf = abs(Xf(1:nf+1));
    if nf+1 > 2
        Sf(2:end-1) = 2*Sf(2:end-1);
    end

    % Normalize
    Sx = Sx / (max(Sx) + eps);
    Sf = Sf / (max(Sf) + eps);

    % Accumulate
    Sraw_sum   = Sraw_sum   + Sx;
    Sraw_sumsq = Sraw_sumsq + Sx.^2;
    Sf_sum     = Sf_sum     + Sf;
    Sf_sumsq   = Sf_sumsq   + Sf.^2;
end

%% --- Mean and std ---
Sraw_mean = Sraw_sum / nPairs;
Sf_mean   = Sf_sum   / nPairs;

Sraw_std  = sqrt(max(Sraw_sumsq / nPairs - Sraw_mean.^2, 0));
Sf_std    = sqrt(max(Sf_sumsq   / nPairs - Sf_mean.^2,   0));

%% --- Figure: (a) Before, (b) Filter, (c) After ---
figure('Color','w','Position',[100 100 1200 700]);
tl = tiledlayout(3,1,'TileSpacing','compact','Padding','compact');
title(tl, sprintf('Mean Amplitude Spectrum over all %d pairs —  Before / Filter / After\n(Gaussian multiple band-pass 2–100 Hz, \\sigma = %.2f)', nPairs, sigma));

% (a) BEFORE
ax1 = nexttile;
plot_mean_std_log(f, Sraw_mean, Sraw_std, [0 0 0]);
xlim([0 100]); ylim([1e-4 1]);
ylabel('Amplitude (norm., log)');
title('Before filtering (mean ± std)');
% ← Метка слева
annotation('textbox',[0.01 0.9 0.05 0.05],'String','(a)',...
    'FontWeight','bold','FontSize',14,'EdgeColor','none');

% (b) FILTER
ax2 = nexttile; hold(ax2,'on'); box(ax2,'on'); grid(ax2,'on')
plot(f, Wpos, 'k', 'LineWidth', 1.3);
xlim([0 100]); ylim([0 1.05]);
ylabel('Amplitude');
title('Composite Gaussian band-pass (2–100 Hz)');
annotation('textbox',[0.01 0.6 0.05 0.05],'String','(b)',...
    'FontWeight','bold','FontSize',14,'EdgeColor','none');

% (c) AFTER
ax3 = nexttile;
plot_mean_std_log(f, Sf_mean, Sf_std, [0 0 0]);
xlim([0 100]); ylim([1e-4 1]);
xlabel('Frequency (Hz)'); ylabel('Amplitude (norm., log)');
title('After filtering (mean ± std)');
annotation('textbox',[0.01 0.3 0.05 0.05],'String','(c)',...
    'FontWeight','bold','FontSize',14,'EdgeColor','none');


%% --- Local function must go last ---
function plot_mean_std_log(f, meanv, stdv, lineColor)
    epsy = 1e-5;
    lo = max(meanv - stdv, epsy);
    hi = max(meanv + stdv, epsy);
    hold on; box on; grid on
    px = [f; flipud(f)];
    py = [lo; flipud(hi)];
    fill(px, py, [0.85 0.85 0.85], 'EdgeColor','none', 'FaceAlpha',0.6);
    h  = plot(f, meanv, 'Color', lineColor, 'LineWidth', 1.3);
    set(gca,'YScale','log');
    legend([h], {'Mean ± 1 std'}, 'Location','southwest');
end
