function make_vs_surfaces_point_cells_supersampled_box()
% Vs maps from .fig points:
% - cell = point, без заливки дыр
% - глобальная маска "ever": где не было точек ни в одном слое — всегда NaN
% - апсемплинг 3x + мягкое сглаживание внутри маски
% - единая цветовая шкала
% - оси в метрах от юго-западного угла
% - ДОБАВЛЕНО: центральный квадрат 5×5 м из receivers.xlsx, рисуется на слоях 1..3

% --------- ПУТИ И ФАЙЛЫ ---------
baseDir    = "D:\Рабочий стол\MastersThesis\Figs\Unconstrained";
figFiles   = ["Vs_unconstrained_1m.fig","Vs_unconstrained_2m.fig","Vs_unconstrained_3m.fig","Vs_unconstrained_4m.fig","Vs_unconstrained_5m.fig"];
layerNames = ["0–1 m","1–2 m","2–3 m","3–4 m","4–5 m"];
outName    = "Vs_surfaces_point_cells_SUPERSAMPLED_BOX_UNCONSTRAINED";

% файл ресиверов для определения центрального квадрата
rcvFile    = "D:\Рабочий стол\MastersThesis\receivers.xlsx";
rcvSheet   = "Sheet 1 - latlon_to_utm";

% --------- ВИЗУАЛ ---------
USE_GLOBAL_CLIM = true;             
GLOBAL_CLIM     = [50 300];
upsampleFactor  = 10;                
smoothSigma     = 2.3;              
smoothRadius    = 7;

% --------- 1) ЧТЕНИЕ ТОЧЕК ИЗ .FIG ---------
layers(1:numel(figFiles)) = struct('X',[],'Y',[],'V',[]);
allX=[]; allY=[]; allV=[];
for k=1:numel(figFiles)
    fpath = fullfile(baseDir, figFiles(k));
    if ~isfile(fpath), error("File not found: %s", fpath); end
    openfig(fpath,'invisible'); ax=gca;
    sc = findobj(ax,'Type','Scatter');

    Xk=[]; Yk=[]; Vk=[];
    for s=1:numel(sc)
        xs = sc(s).XData(:);
        ys = sc(s).YData(:);
        cd = sc(s).CData;
        if isnumeric(cd) && isvector(cd) && numel(cd)==numel(xs)
            Xk = [Xk; xs];  Yk = [Yk; ys];  Vk = [Vk; cd(:)];
        end
    end
    close(gcf);
    layers(k).X=Xk; layers(k).Y=Yk; layers(k).V=Vk;
    allX=[allX; Xk]; allY=[allY; Yk]; allV=[allV; Vk];
end
if isempty(allX), error('No colored Vs points found in the .fig files.'); end

% --------- 2) СЕТКА (CELL=POINT) ---------
Ux = unique(allX(:)); Uy = unique(allY(:));
xEdges = midEdges(Ux); yEdges = midEdges(Uy);
xCenters = Ux(:).'; yCenters = Uy(:).';

maps  = cell(1,numel(layers)); masks = cell(1,numel(layers));
for k=1:numel(layers)
    X=layers(k).X; Y=layers(k).Y; V=layers(k).V;
    ix = discretize(X, xEdges);
    iy = discretize(Y, yEdges);
    G = nan(numel(Uy), numel(Ux));
    for i=1:numel(V)
        if ~isnan(ix(i)) && ~isnan(iy(i))
            G(iy(i),ix(i)) = V(i);
        end
    end
    maps{k}  = G; 
    masks{k} = ~isnan(G);
end

% --------- 3) ГЛОБАЛЬНАЯ МАСКА ---------
everMask = false(size(masks{1}));
for k=1:numel(masks), everMask = everMask | masks{k}; end

% --------- 4) UPSAMPLING (nearest) + SMOOTHING ВНУТРИ МАСКИ ---------
K = gaussKernel(smoothSigma, smoothRadius);
mapsUp = cell(1,numel(maps)); everUp = [];

for k=1:numel(maps)
    G = maps{k};     M = masks{k};
    Gup = imresize(G, upsampleFactor, 'nearest');
    Mup = imresize(M, upsampleFactor, 'nearest');
    Gup(~Mup) = NaN;

    num = conv2(replaceNaN(Gup,0), K, 'same');
    den = conv2(double(Mup),       K, 'same');
    Gup = num ./ max(den, eps);
    Gup(~Mup) = NaN;

    if isempty(everUp), everUp = imresize(everMask, size(Gup), 'nearest'); end
    Gup(~everUp) = NaN;

    mapsUp{k} = Gup;
end

% --------- 5) ЧИТАЕМ ПОЗИЦИЮ ЦЕНТРАЛЬНОГО КВАДРАТА 5×5 м ИЗ receivers.xlsx ---------
roi = [];
try
    data = readtable(rcvFile,'Sheet',rcvSheet);
    data = data(2:end,:);                    % пропустить заголовочную строку-единицы
    data.Properties.VariableNames = {'Name','X','Y'};
    RX = str2double(data.X);
    RY = str2double(data.Y);

    nbins = 60;
    [Ny,ey] = histcounts(RY, nbins); [~,iy] = max(Ny);
    y0m = median(RY(RY>=ey(iy) & RY<ey(iy+1)));
    [Nx,ex] = histcounts(RX, nbins); [~,ix] = max(Nx);
    x0m = median(RX(RX>=ex(ix) & RX<ex(ix+1)));

    halfSide = 2.5;     % половина стороны (м)
    dx_shift = -1.2;    % сдвиг по X (м), как в твоём коде
    roi = struct('x', x0m-halfSide+dx_shift, ...
                 'y', y0m-halfSide, ...
                 'w', 2*halfSide, ...
                 'h', 2*halfSide);
catch
    roi = [];
end
hasROI = ~isempty(roi);

% --------- 6) ОСИ: в метрах от ЮЗ-угла ---------
x0 = min(xCenters); y0 = min(yCenters);
xRange = max(xCenters) - x0;  yRange = max(yCenters) - y0;
xStep = niceStep(xRange, 5);  yStep = niceStep(yRange, 5);

% --------- 7) ОТРИСОВКА (3 сверху, 2 снизу, общий colorbar снизу) ---------
F = figure('Color','w','Position',[100 100 1500 780]);

% 2 строки × 3 колонки (6-я плитка останется пустой), colorbar в "south"
tl = tiledlayout(F, 2, 3, 'TileSpacing','compact', 'Padding','compact');

for k = 1:numel(mapsUp)   % k = 1..5
    nexttile(tl, k);      % заполнит: [1 2 3] на 1-й строке, [4 5] на 2-й
    G = mapsUp{k};

    xPlot = linspace(0, xRange, size(G,2));
    yPlot = linspace(0, yRange, size(G,1));

    imagesc(xPlot, yPlot, G, 'AlphaData', ~isnan(G));
    set(gca,'YDir','normal','Color','none');
    axis image tight; box on; grid off;

    xticks(0:xStep:ceil(xRange/xStep)*xStep);
    yticks(0:yStep:ceil(yRange/yStep)*yStep);
    xlabel('X (m)'); ylabel('Y (m)');
    title(layerNames(k),'FontWeight','bold');

    colormap(turbo);
    if USE_GLOBAL_CLIM
        caxis(GLOBAL_CLIM);
    else
        [vmin,vmax] = safeMinMax(G, min(allV), max(allV));
        caxis([vmin vmax]);
    end

    % --- Квадрат только для первых трёх слоёв ---
    if hasROI && k<=3
        rx = roi.x - x0;  ry = roi.y - y0;
        rectangle('Position',[rx, ry, roi.w, roi.h], ...
                  'EdgeColor',[0 0 0], 'LineWidth',1.6, 'LineStyle','--');
    end
end

% Общий colorbar внизу под всей сеткой
cb = colorbar('Location','southoutside');
cb.Label.String = 'V_S (m/s)';
cb.Layout.Tile  = 'south';

% Сохранение
savefig(F, fullfile(baseDir, outName + ".fig"));
exportgraphics(F, fullfile(baseDir, outName + ".png"), 'Resolution', 400);
fprintf('Saved: %s\n', fullfile(baseDir, outName + ".png"));

end

% -------------------- HELPERS --------------------
function edges = midEdges(centers)
centers = centers(:);
if numel(centers)==1, d=1; edges=[centers(1)-d; centers(1)+d]; return; end
mids = (centers(1:end-1)+centers(2:end))/2;
firstEdge = centers(1) - (mids(1)-centers(1));
lastEdge  = centers(end) + (centers(end)-mids(end));
edges = [firstEdge; mids; lastEdge];
end

function A = replaceNaN(A,val)
A(isnan(A)) = val;
end

function K = gaussKernel(sigma, radius)
radius = max(1, ceil(radius));
[x,y] = meshgrid(-radius:radius, -radius:radius);
K = exp(-(x.^2 + y.^2)/(2*sigma^2));
K = K / sum(K(:));
end

function [vmin,vmax] = safeMinMax(A, fallbackMin, fallbackMax)
v = A(~isnan(A));
if isempty(v), vmin=fallbackMin; vmax=fallbackMax; return; end
vmin = min(v); vmax = max(v);
if ~isfinite(vmin) || ~isfinite(vmax) || vmin==vmax
    vmin=fallbackMin; vmax=fallbackMax;
end
end

function step = niceStep(rangeVal, targetNTicks)
if rangeVal<=0, step=1; return; end
raw = rangeVal / max(3, targetNTicks);
pow10 = 10.^floor(log10(raw));
frac = raw ./ pow10;
if frac < 1.5, nice=1;
elseif frac < 3, nice=2;
elseif frac < 7, nice=5;
else, nice=10;
end
step = nice .* pow10;
end
