stats_cons = analyze_fig_fit_v2('D:\Рабочий стол\MastersThesis\Figs\Constrained\fitting_dc.fig','fit_cons');
stats_un   = analyze_fig_fit_v2('D:\Рабочий стол\MastersThesis\Figs\Unconstrained\fitting_dc_un.fig','fit_uncons');
function stats = analyze_fig_fit_v2(figPath, outPrefix, opts)
% ANALYZE_FIG_FIT_V2
% Оценивает fit синтетических (красных) дисперсионных кривых к наблюдённым (чёрным)
% прямо из .fig файла.
%
% ВЫХОД:
%   stats: struct с полями RMSE_mps, MAE_mps, MedianAE_mps, P90AE_mps, NormMisfit_pct, N
% Сохранения:
%   [outPrefix]_pointwise.csv     - по-точечные ошибки (f, c_obs, c_syn, Δc, |Δc|)
%   [outPrefix]_summary.csv       - сводные метрики
%   [outPrefix]_hist.png          - гистограмма |Δc|
%   [outPrefix]_freq_misfit.png   - |Δc| по частоте (среднее и P90)

    if nargin < 3 || ~isstruct(opts), opts = struct(); end
    df_tol   = getdef(opts,'df_tol_hz', 0.25);     % допуск частоты для fallback NN
    f_min    = getdef(opts,'f_min', -inf);         % ограничение диапазона частот (Hz)
    f_max    = getdef(opts,'f_max',  inf);
    col_tol  = getdef(opts,'color_tol', 0.03);     % допуск сравнения RGB
    rgb_obs  = getdef(opts,'rgb_obs', [0 0 0]);    % цвет наблюдений (чёрный)
    rgb_syn  = getdef(opts,'rgb_syn', [1 0 0]);    % цвет синтетики (красный)

    % --- открыть фигуру невидимо
    fig = openfig(figPath,'invisible'); c = onCleanup(@() close(fig));
    ax  = findall(fig,'type','axes');
    lns = findall(ax,'type','line');
    scs = findall(ax,'type','scatter');

    % --- собрать точки (и line, и scatter)
    [Fx, Cx] = collect_lines(lns, rgb_obs, col_tol);   % наблюдения
    [Fr, Cr] = collect_lines(lns, rgb_syn, col_tol);   % синтетика
    [Fx2,Cx2]= collect_scatter(scs, rgb_obs, col_tol);
    [Fr2,Cr2]= collect_scatter(scs, rgb_syn, col_tol);

    Fx=[Fx;Fx2]; Cx=[Cx;Cx2];
    Fr=[Fr;Fr2]; Cr=[Cr;Cr2];

    % --- очистка/фильтр по диапазону
    mX = isfinite(Fx) & isfinite(Cx) & Fx>=f_min & Fx<=f_max;
    mR = isfinite(Fr) & isfinite(Cr) & Fr>=f_min & Fr<=f_max;
    Fx = Fx(mX); Cx = Cx(mX);
    Fr = Fr(mR); Cr = Cr(mR);

    if isempty(Fx) || isempty(Fr)
        error('Не нашёл красные и/или чёрные точки в "%s". Проверь цвета или объекты.', figPath);
    end

    % --- «устойчивый» интерполянт синтетики: среднее по уникальным частотам
    [Fr,ord] = sort(Fr(:)); Cr = Cr(ord);
    [Fu,~,ic] = unique(Fr);      % ic >= 1 — безопасно для accumarray
    Cs = accumarray(ic, Cr, [], @mean);

    % интерполяция синтетики на частоты наблюдений (с экстраполяцией)
    Csyn = interp1(Fu, Cs, Fx, 'linear', 'extrap');

    % fallback: ближайший сосед, если вдруг что-то осталось NaN/Inf
    nanI = isnan(Csyn) | ~isfinite(Csyn);
    if any(nanI)
        for k = find(nanI).'
            [mindf,j] = min(abs(Fu - Fx(k)));
            if ~isempty(mindf) && mindf <= df_tol
                Csyn(k) = Cs(j);
            end
        end
    end

    % валидная маска
    valid = isfinite(Csyn) & isfinite(Cx);
    Fx = Fx(valid); Cx = Cx(valid); Csyn = Csyn(valid);

    % --- метрики
    res   = Csyn - Cx;           % синтетика - наблюдение (м/с)
    absr  = abs(res);
    RMSE  = sqrt(mean(res.^2));
    MAE   = mean(absr);
    MED   = median(absr);
    P90   = prctile(absr, 90);
    Norm  = 100 * RMSE / mean(Cx);

    % --- сохранение CSV
    T = table(Fx, Cx, Csyn, res, absr, ...
        'VariableNames', {'f_Hz','c_obs_mps','c_syn_mps','res_mps','absres_mps'});
    writetable(T, [outPrefix '_pointwise.csv']);

    S = table(RMSE, MAE, MED, P90, Norm, ...
        'VariableNames', {'RMSE_mps','MAE_mps','MedianAE_mps','P90AE_mps','NormMisfit_pct'});
    writetable(S, [outPrefix '_summary.csv']);

    % --- отчётные графики
    figure('Color','w');
    histogram(absr, 60); grid on
    xlabel('|Δc| (m/s)'); ylabel('Count');
    title(['Abs residuals: ' outPrefix], 'Interpreter','none');
    saveas(gcf,[outPrefix '_hist.png']);

    [fgrid, mu_abs, p90_abs] = freq_stats(Fx, absr, 1.0);  % шаг 1 Гц
    figure('Color','w');
    plot(fgrid, mu_abs, '-', 'LineWidth', 1.6); hold on
    plot(fgrid, p90_abs,'--','LineWidth', 1.2); grid on
    xlabel('Frequency (Hz)'); ylabel('Abs error (m/s)');
    legend('Mean |res|','P90 |res|','Location','best');
    title(['Freq-dependent misfit: ' outPrefix], 'Interpreter','none');
    saveas(gcf,[outPrefix '_freq_misfit.png']);

    % --- вывод
    stats = struct('RMSE_mps',RMSE,'MAE_mps',MAE,'MedianAE_mps',MED, ...
                   'P90AE_mps',P90,'NormMisfit_pct',Norm,'N',numel(Fx));
    fprintf('[%s] RMSE=%.2f  MAE=%.2f  median=%.2f  P90=%.2f  Norm=%.1f%%  N=%d\n', ...
            outPrefix, RMSE, MAE, MED, P90, Norm, numel(Fx));
end

% ======================= helpers ===========================
function v = getdef(s, fn, def)
    if isfield(s,fn), v = s.(fn); else, v = def; end
end

function tf = closeRGB(a, b, tol)
    if nargin < 3, tol = 0.03; end
    tf = isnumeric(a) && numel(a)==3 && all(abs(a(:)-b(:)) <= tol);
end

function [F,C] = collect_lines(lns, rgb, tol)
    if nargin < 3, tol = 0.03; end
    F = []; C = [];
    for k = 1:numel(lns)
        col = []; 
        try, col = get(lns(k),'Color'); end %#ok<TRYNC>
        if isempty(col) || ~isnumeric(col), continue; end
        if closeRGB(col, rgb, tol)
            x = get(lns(k),'XData'); 
            y = get(lns(k),'YData');
            m = isfinite(x) & isfinite(y);
            x = x(m);  y = y(m);
            F = [F; x(:)];
            C = [C; y(:)];
        end
    end
end

function [F,C] = collect_scatter(scs, rgb, tol)
    if nargin < 3, tol = 0.03; end
    F = []; C = [];
    for k = 1:numel(scs)
        % пробуем MarkerEdgeColor, если 'flat' — цвет берётся из CData, пропускаем
        col = [NaN NaN NaN];
        try
            col = get(scs(k),'MarkerEdgeColor');
            if ischar(col) && strcmpi(col,'flat'), col = [NaN NaN NaN]; end
        end %#ok<TRYNC>
        if ~any(isnan(col)) && closeRGB(col, rgb, tol)
            x = get(scs(k),'XData'); 
            y = get(scs(k),'YData');
            m = isfinite(x) & isfinite(y);
            x = x(m);  y = y(m);
            F = [F; x(:)];
            C = [C; y(:)];
        end
    end
end

function [fgrid, mu, p90] = freq_stats(f, absr, step_hz)
    if nargin < 3, step_hz = 1.0; end
    fmin = floor(min(f));
    fmax = ceil(max(f));
    fgrid = (fmin:step_hz:fmax).';
    mu   = nan(size(fgrid));
    p90  = nan(size(fgrid));
    half = step_hz/2;
    for i = 1:numel(fgrid)
        in = abs(f - fgrid(i)) <= half;
        if any(in)
            mu(i)  = mean(absr(in));
            p90(i) = prctile(absr(in), 90);
        end
    end
end
