%% Batch_Filter_AllPairs.m
% Применение фильтрации 2–100 Hz ко всем парам ресиверов (CC_mat)
% --------------------------------------------------------------
% Input:  CNR_CC_mat.mat (матрица CC_mat [nt x nPairs])
% Output: QC-графики усреднённых спектров и RMS-энергии по всем парам

clear; close all; clc

%% ------------------ PARAMETERS ------------------
fmin  = 2;               % нижняя граница (Hz)
fmax  = 100;             % верхняя граница (Hz)
df0   = 0.5;             % шаг по частоте для банка (Hz)
sigma = 0.15;            % относительная ширина Гаусса
dt    = 0.001;           % шаг по времени (s)
fs    = 1/dt;

%% ------------------ LOAD DATA ------------------
load('CNR_CC_mat.mat');   % должен содержать CC_mat [nt x nPairs]
[nt, nPairs] = size(CC_mat);
t = (0:nt-1)' * dt;

fprintf('Loaded %d pairs, %d samples per trace\n', nPairs, nt);

%% ------------------ BUILD GAUSSIAN FILTER ------------------
nf = floor(nt/2);
f  = (0:nf)' * (fs/nt);
f0 = fmin:df0:fmax;

% Банк гауссов
G = exp(-0.5 * ((f - f0) ./ (sigma .* f0)).^2);
Wpos = max(G, [], 2);
Wpos(f < fmin) = 0;
Wpos(f > fmax) = 0;

% Симметричный H(f)
H = zeros(nt,1);
H(1:nf+1) = Wpos;
if mod(nt,2)==0
    H(nf+2:end) = Wpos(end-1:-1:2);
else
    H(nf+2:end) = Wpos(end:-1:2);
end

%% ------------------ PROCESS ALL PAIRS ------------------
E_before = zeros(nPairs,1);
E_after  = zeros(nPairs,1);
S_before = zeros(nf+1, nPairs);
S_after  = zeros(nf+1, nPairs);

for i = 1:nPairs
    x = CC_mat(:,i);
    x = detrend(x - mean(x));

    % FFT
    X = fft(x);
    S = abs(X(1:nf+1));  S(2:end-1) = 2*S(2:end-1);
    S_before(:,i) = S / (max(S) + eps);

    % Filtering
    Xf = X .* H;
    xf = real(ifft(Xf));

    Sf = abs(Xf(1:nf+1));  Sf(2:end-1) = 2*Sf(2:end-1);
    S_after(:,i) = Sf / (max(Sf) + eps);

    % Energy (RMS)
    E_before(i) = rms(x);
    E_after(i)  = rms(xf);
end

%% ------------------ QC: ENERGY RATIO ------------------
ratio = E_after ./ (E_before + eps);

fprintf('Average energy ratio after filtering: %.3f ± %.3f\n', mean(ratio), std(ratio));

%% ------------------ PLOT AVERAGED SPECTRA ------------------
figure('Color','w');
plot(f, mean(S_before,2), '--', 'Color',[0.7 0 0], 'LineWidth',1.2); hold on
plot(f, mean(S_after,2),  '-',  'Color','k',       'LineWidth',1.4);
set(gca,'YScale','log'); xlim([0 200]); ylim([1e-4 1]);
grid on; box on;
xlabel('Frequency (Hz)');
ylabel('Mean amplitude (log scale)');
legend('Before filtering','After filtering','Location','southwest');
title('Average amplitude spectrum across all receiver pairs');

%% ------------------ PLOT ENERGY RATIO HISTOGRAM ------------------
figure('Color','w');
histogram(ratio, 30, 'FaceColor',[0.4 0.4 0.4]);
xlabel('E_{after} / E_{before}');
ylabel('Count');
title('Distribution of RMS energy ratio after filtering');
grid on; box on;

%% ------------------ PLOT FILTER RESPONSE ------------------
figure('Color','w');
plot(f, Wpos, 'k', 'LineWidth',1.3);
xlabel('Frequency (Hz)');
ylabel('Amplitude');
title(sprintf('Composite Gaussian band-pass filter (%.1f–%.1f Hz, σ=%.2f)', fmin, fmax, sigma));
xlim([0 200]); ylim([0 1.05]); grid on; box on;

%% ------------------ SAVE (optional) ------------------
% save('Filtered_AllPairs.mat','E_before','E_after','ratio','S_before','S_after','f');
