%% ===== Receiver Paths Map (only SELECTED pairs) =====
clear; close all; clc

data_dir = 'D:\Рабочий стол\MastersThesis\DC_interferometry';
S = dir(fullfile(data_dir,'*_r2_*.mat'));

% Настройки визуализации
AZ_EDGES = 0:15:180;         % бинning азимута (как в розе)
MAX_DRAW = inf;              % при желании: ограничить число рисуемых путей
LINE_W   = 0.8;              % толщина линий
EDGE_ALF = 0.15;             % прозрачность линий

% Аккумуляторы
pairs   = [];    % [x1 y1 x2 y2 az_bin]
nodesXY = [];    % все узлы (для scatter)

count_sel = 0;
for k = 1:numel(S)
    T = load(fullfile(S(k).folder, S(k).name));

    % берем только "выбранные" пары (валидные pick'и)
    if ~isfield(T,'stack_dc') || ~isfield(T,'freqtot'), continue; end
    f = T.freqtot(:);  c = T.stack_dc(:);
    ok = isfinite(f) & isfinite(c) & f>0 & c>0;
    if nnz(ok) < 3, continue; end

    % координаты пары
    if ~isfield(T,'coordinates') || numel(T.coordinates) < 4, continue; end
    xy = T.coordinates(:).';
    x1 = xy(1); y1 = xy(2); x2 = xy(3); y2 = xy(4);

    % азимут (0..180°, ориентация)
    dX = x2 - x1;  dY = y2 - y1;
    az = atan2d(dX, dY); if az < 0, az = az + 360; end
    if az > 180, az = az - 180; end

    % бин по азимуту
    [~,~,b] = histcounts(az, AZ_EDGES);
    if b==0, b=1; end

    pairs(end+1,:) = [x1 y1 x2 y2 b]; %#ok<AGROW>
    nodesXY = [nodesXY; x1 y1; x2 y2]; %#ok<AGROW>

    count_sel = count_sel + 1;
    if count_sel >= MAX_DRAW, break; end
end

fprintf('Selected pairs used for map: %d\n', size(pairs,1));
if isempty(pairs)
    error('Нет валидных пар. Проверь файлы *_r2_*.mat (stack_dc, freqtot, coordinates).');
end

% Палитра по биннам
nBins = max(pairs(:,5));
cmap  = turbo(max(nBins,12));

% Фигура
figure('Color','w','Position',[80 80 900 700]); hold on; box on; grid on
axis equal

% (1) Узлы (приёмники)
% Уберём дубликаты
[uniqXY, ~] = unique(nodesXY, 'rows', 'stable');
scatter(uniqXY(:,1), uniqXY(:,2), 18, 'k', 'filled', 'MarkerFaceAlpha', 0.8);

% (2) Пути (линии) с прозрачностью через patch (line не умеет alpha)
for b = 1:nBins
    idx = (pairs(:,5)==b);
    if ~any(idx), continue; end
    segs = pairs(idx,1:4);
    % Векторизованная отрисовка пачкой:
    % сделаем массивы X,Y c NaN-разделителями, чтобы одним patch нарисовать много сегментов
    X = [segs(:,1) segs(:,3) nan(size(segs,1),1)].';  % [x1; x2; NaN] по столбцам
    Y = [segs(:,2) segs(:,4) nan(size(segs,1),1)].';
    X = X(:); Y = Y(:);
    ph = patch(X, Y, 'w', ...
        'EdgeColor', cmap(b,:), 'FaceColor','none', ...
        'LineWidth', LINE_W, 'EdgeAlpha', EDGE_ALF);
end

% Рамка и подписи
xlabel('Easting / X (m)'); ylabel('Northing / Y (m)');
title('Receiver Paths Map (selected dispersion pairs, colored by azimuth)');

% Легенда-цветовая шкала по азимуту (создадим “псевдо” colorbar)
colormap(cmap);
cb = colorbar('Location','eastoutside'); 
cb.Ticks = linspace(0,1,numel(AZ_EDGES));
cb.TickLabels = arrayfun(@(a)sprintf('%d°',a), AZ_EDGES, 'uni',0);
cb.Label.String = 'Pair azimuth (0–180°)';

% Пределы карты с небольшим полем
pad = 0.02;
minx = min(uniqXY(:,1)); maxx = max(uniqXY(:,1));
miny = min(uniqXY(:,2)); maxy = max(uniqXY(:,2));
dx = maxx - minx; dy = maxy - miny;
xlim([minx - pad*dx, maxx + pad*dx]);
ylim([miny - pad*dy, maxy + pad*dy]);

set(gca,'FontName','Times New Roman','FontSize',12,'LineWidth',1,'Layer','top');

% Сохранить при необходимости
% exportgraphics(gcf, fullfile(data_dir,'Receiver_Paths_Map_selected.png'), 'Resolution', 300);
