from matplotlib import pyplot as plt
from time import sleep
import TimeTagger
import numpy as np

# --- Time Tagger Hardware Initialization and Configuration ---

# Create a Time Tagger instance to control the hardware
tagger = TimeTagger.createTimeTagger()

# Define the total measurement duration (in seconds)
total_measurement_time_s = 120  # Example: 120 s

# Define the acquisition window size for each iteration (in seconds)
acquisition_window_s = 0.1  # Example: 100 ms

# Set the DESIRED binwidth (in picoseconds)
# Note: 100 ms = 100 * 10^6 ps
binwidth_ps = int(100e6)  # 100 ms (FIXED resolution)

# Calculate the number of iterations (based on total duration and acquisition window)
num_iterations = int(total_measurement_time_s / acquisition_window_s)

# Calculate the maximum number of bins needed to store data for the total duration
max_bins = int(total_measurement_time_s / (binwidth_ps / 1e12))

# Create a Counter instance ONCE with the predefined resolution
# Channel [2] is used for counting (adjust channel number as needed)
counter = TimeTagger.Counter(tagger=tagger, channels=[2], binwidth=binwidth_ps, n_values=max_bins)

# --- Dynamic Plot Setup (Matplotlib) ---

# Create the figure and axes for the dynamic plot
fig, ax = plt.subplots()
# Assuming Channel 2 corresponds to 'Channel 1' in the plot label
line, = ax.plot([], [], label='Channel 1 (External Trigger)') 
ax.set_xlabel('Time ($s$)')
# Dynamic label showing the bin resolution (100 ms in this case)
ax.set_ylabel(f'Counts per {binwidth_ps / 1e9 * 1e3} $ms$') 
ax.set_title(f'Counts over Time (Resolution {binwidth_ps / 1e9 * 1e3} $ms$) - Channel 2')
ax.grid(True)
ax.legend()

# Initialize lists to store acquired data
all_time_data = np.array([])
all_count_data = np.array([])

plt.ion()  # Enable interactive mode for dynamic plot updates
plt.show()

# --- Data Acquisition Loop ---

try:
    for i in range(num_iterations):
        print(f"Acquiring data segment {i+1}/{num_iterations}...")
        
        # Wait for the specified acquisition window duration
        sleep(acquisition_window_s)

        # Retrieve the cumulative measured data up to this point
        data = counter.getData()
        data_channel1 = data[0]

        # Get the corresponding time axis for ALL acquired bins (in seconds)
        time_axis = counter.getIndex() / 1e12  # in seconds

        # Update the data arrays
        all_time_data = time_axis
        all_count_data = data_channel1

        # Update the plot line data
        line.set_xdata(all_time_data)
        line.set_ydata(all_count_data)

        # Update axes limits dynamically
        ax.relim()
        ax.autoscale_view()

        # Redraw and flush the plot canvas
        fig.canvas.draw()
        fig.canvas.flush_events()

        sleep(0.1) # Short pause between readouts
        
except KeyboardInterrupt:
    print("Acquisition interrupted by user.")

finally:
    # Stop the counter process
    counter.stop()
    # Release the Time Tagger instance
    TimeTagger.freeTimeTagger(tagger)
    
    # Disable interactive mode and show the final plot
    plt.ioff()
    plt.show()