# -*- coding: utf-8 -*-
#
# This Python script configures and controls the FPGA (Field-Programmable Gate Array) boards
# to generate specific intensity and phase modulation patterns.
# The main objective is the tuning of fine and coarse delays and the modulation of the pattern,
# which are fundamental aspects for Quantum Key Distribution (QKD) experiments.
#
# The code connects to the FPGA hardware, maps the memory registers for controlling the DACs (Digital-to-Analog Converters)
# and loads predefined patterns into the board's memory for playback.
#

# Main configuration section
# Import PYNQ packages for interacting with the FPGA board
from xrfdc import *
from xrfclk import *
from pynq import MMIO             # Access to system memory
from pynq import Overlay           # Loads the hardware description of the SoC system on the board
import matplotlib.pyplot as plt
import numpy as np

# Load the bitstream file (hardware configuration file) onto the FPGA.
# This file defines the architecture of the logic circuit on the board.
# The `ignore_version=True` flag allows loading even if the bitstream version does not match.
base = Overlay("/home/xilinx/bit_qkd/design_1_wrapper_8ch_delayb.bit", ignore_version = True)
base.download()

# Set the reference clock frequency for the RF (Radio Frequency) components
# LMK is the clock generator, LMX is the frequency synthesizer
set_ref_clks(lmk_freq=500, lmx_freq=4000)

# Reload the bitstream to ensure the clock is stable.
# This is a common practice to prevent timing issues.
base = Overlay("/home/xilinx/bit_qkd/design_1_wrapper_8ch_delayb.bit", ignore_version = True)
base.download()


# Main system memory map
# All memory-mapped devices inserted in the VHDL code are defined here.
# This allows the Python software to write to and read from specific hardware registers.
# `adc_cap` and `dac_play` are the addresses for the capture and playback registers.

# Memory addresses for ADC acquisition
adc_cap0 = 0x00A0200000
adc_cap1 = 0x00A0280000
adc_cap2 = 0x00A0300000
adc_cap3 = 0x00A0380000
dac_cap0 = 0x00A0180000

# Memory addresses for DAC playback
dac_play = 0x00A0100000
dac_play0 = 0x00A0100000
dac_play1 = 0x00A0400000
dac_play2 = 0x00A0480000
dac_play3 = 0x00A0500000
dac_play4 = 0x00A0580000
dac_play5 = 0x00A0600000
dac_play6 = 0x00A0680000
dac_play7 = 0x00A0700000

# GPIO (General-Purpose Input/Output) control register addresses
gpio_adc = 0x00A0060000
dac_ctrl = 0x00A00B0000
gpio_dac = 0x00A0043000
gpio_1_out = 0xA0090000
pps_gen = 0xA00A_0000

# Memory size for acquisition/playback
mem_size_cap = 2**17

# Create MMIO objects for each address, allowing
# access to the memory-mapped devices.
adc_cap0_mmio = MMIO(adc_cap0,mem_size_cap)
adc_cap1_mmio = MMIO(adc_cap1,mem_size_cap)
adc_cap2_mmio = MMIO(adc_cap2,mem_size_cap)
adc_cap3_mmio = MMIO(adc_cap3,mem_size_cap)
dac_cap0_mmio = MMIO(dac_cap0,mem_size_cap)
dac_play0_mmio = MMIO(dac_play0,mem_size_cap)
dac_play1_mmio = MMIO(dac_play1,mem_size_cap)
dac_play2_mmio = MMIO(dac_play2,mem_size_cap)
dac_play3_mmio = MMIO(dac_play3,mem_size_cap)
dac_play4_mmio = MMIO(dac_play4,mem_size_cap)
dac_play5_mmio = MMIO(dac_play5,mem_size_cap)
dac_play6_mmio = MMIO(dac_play6,mem_size_cap)
dac_play7_mmio = MMIO(dac_play7,mem_size_cap)


mem_size_regs = 1024
base_gpio_adc = MMIO(gpio_adc, mem_size_regs)
base_gpio_dac = MMIO(gpio_dac, mem_size_regs)
base_gpio_ctr = MMIO(dac_ctrl, mem_size_regs)
base_gpio_sma = MMIO(gpio_1_out, mem_size_regs)
base_pps_gen = MMIO(pps_gen, mem_size_regs)

# REGISTER INITIALIZATION
# ALICE = a
dac_a_IM1 = 0
dac_a_IM2 = 12
dac_a_PM = 24
# BOB = b
dac_b_IM1 = 48
dac_b_IM2 = 60
dac_b_PM = 72

# FINE DELAY REGISTER INITIALIZATION
# ALICE = a
df_a_IM1 = 24
df_a_IM2 = 25
df_a_PM = 26
# BOB = b
df_b_IM1 = 28
df_b_IM2 = 29
df_b_PM = 30

"""Reference values for the DAC output levels
verified with ixblue drivers - best linearity with 110mV output
the output tends to saturate with higher values

for 180mVpp
a= 0xD800;       otherwise:        a=0x37FF;


for 110mV
a= 0xE800;       otherwise:        a=0x17FF;


for 390mVpp
a= 0x8800;       otherwise:          a=0x77FF;

"""

# CREATE THE DESIRED PATTERN
# 250MHz square wave pattern with 0
w = np.int32(0)
w1 = np.int32(0)
sent_data = np.zeros(4*8192, dtype = np.int32)

# HERE YOU CAN SET THE DESIRED LEVEL VALUES
for i in range(0, 8192):
    if i % 2 == 0:
        a = 0xE800
    else:
        a = 0x17FF

    if i % 4 == 0:
        a = 0xF800
        # a = 0x17FF

# CREATE THE WAVEFORM IN RAM
    w = a
    w = np.int32((np.int32(a) << 16) | w)
    sent_data[i] = w
    
    # Write the pattern into the DAC memory
    # This loop distributes the waveform across all 8 DAC channels.
    dac_play0_mmio.write(i*16, int(w))
    dac_play0_mmio.write(i*16 + 4, int(w))
    dac_play0_mmio.write(i*16 + 8, int(w))
    dac_play0_mmio.write(i*16 + 12, int(w))
    dac_play1_mmio.write(i*16, int(w))
    dac_play1_mmio.write(i*16 + 4, int(w))
    dac_play1_mmio.write(i*16 + 8, int(w))
    dac_play1_mmio.write(i*16 + 12, int(w))
    dac_play2_mmio.write(i*16, int(w))
    dac_play2_mmio.write(i*16 + 4, int(w))
    dac_play2_mmio.write(i*16 + 8, int(w))
    dac_play2_mmio.write(i*16 + 12, int(w))
    dac_play3_mmio.write(i*16, int(w))
    dac_play3_mmio.write(i*16 + 4, int(w))
    dac_play3_mmio.write(i*16 + 8, int(w))
    dac_play3_mmio.write(i*16 + 12, int(w))
    dac_play4_mmio.write(i*16, int(w))
    dac_play4_mmio.write(i*16 + 4, int(w))
    dac_play4_mmio.write(i*16 + 8, int(w))
    dac_play4_mmio.write(i*16 + 12, int(w))
    dac_play5_mmio.write(i*16, int(w))
    dac_play5_mmio.write(i*16 + 4, int(w))
    dac_play5_mmio.write(i*16 + 8, int(w))
    dac_play5_mmio.write(i*16 + 12, int(w))
    dac_play6_mmio.write(i*16, int(w))
    dac_play6_mmio.write(i*16 + 4, int(w))
    dac_play6_mmio.write(i*16 + 8, int(w))
    dac_play6_mmio.write(i*16 + 12, int(w))
    dac_play7_mmio.write(i*16, int(w))
    dac_play7_mmio.write(i*16 + 4, int(w))
    dac_play7_mmio.write(i*16 + 8, int(w))
    dac_play7_mmio.write(i*16 + 12, int(w))

# ENABLE DAC CHANNELS FOR PLAYBACK
# 1 -> enables the channel in continuous mode (free-run)
# 2 -> enables the channel with a trigger (internal or external)
# In this section, registers are written using the variables initialized at the beginning of the file.
base_gpio_ctr.write(dac_a_IM1, 1) # Control of the DAC for Alice's IM1 modulator
base_gpio_ctr.write(dac_a_IM2, 1) # Control of the DAC for Alice's IM2 modulator
base_gpio_ctr.write(dac_a_PM, 0) # Control of the DAC for Alice's PM modulator
# base_gpio_ctr.write(36, 1) # X
base_gpio_ctr.write(dac_b_IM1, 0) # Control of the DAC for Bob's IM1 modulator
base_gpio_ctr.write(dac_b_IM2, 0) # Control of the DAC for Bob's IM2 modulator
base_gpio_ctr.write(dac_b_PM, 0) # Control of the DAC for Bob's PM modulator
# base_gpio_ctr.write(84, 1) # X

# Fine Delay Adjustment
# This section aligns the timing of pulses with a high degree of precision.
# The `base_gpio_ctr` registers for fine delay are calibrated experimentally
# to compensate for small differences in fiber length and component latency.
# The units are typically fractions of a sampling clock period.

# Fine delay for Alice's modulators
base_gpio_ctr.write(df_a_IM1 * 4, 0) # The delay of Alice's IM1 modulator is set to zero as a reference.
base_gpio_ctr.write(df_a_IM2 * 4, 12) # The delay of Alice's IM2 modulator is aligned with IM1. The value is found experimentally.
# Phase
base_gpio_ctr.write(df_a_PM * 4, 5) # The delay of Alice's PM modulator is aligned with IM1 and IM2 by observing the interference pattern.

# Fine delay for Bob's modulators
base_gpio_ctr.write(df_b_IM1 * 4, 24) # The delay of Bob's IM1 modulator is aligned with Alice's IM1 modulator. The value is found experimentally.
base_gpio_ctr.write(df_b_IM2 * 4, 2) # The delay of Bob's IM2 modulator is aligned with Bob's IM1 and Alice's IM1. The value is found experimentally.
# Phase
base_gpio_ctr.write(df_b_PM * 4, 6) # The delay of Bob's PM modulator is aligned with his IMs and with Alice's modulators by observing the interference pattern when both Alice's and Bob's nodes are on.

# Conclusion of fine delay alignment
# After these steps, all four modulators (two for Alice and two for Bob) are aligned for fine delay.

# Coarse Delay Adjustment
# This section controls larger delays, typically to compensate for long fiber distances.
# Coarse delays are set by writing to specific registers and are typically
# synchronized by an external or internal time base.

# to activate and change delays in coarse mode (for long distances)
# an external time base is needed or the internal one must be activated.
# (1)
base_pps_gen.write(4, 165000)
base_pps_gen.write(0, 1)
# (activates the internal one with a period of...)

# (2)
# sets the channels from free-run to reference mode. This is mandatory for coarse delay.
base_gpio_ctr.write(0, 2)
base_gpio_ctr.write(12, 2)
base_gpio_ctr.write(24, 2)
base_gpio_ctr.write(36, 2)
base_gpio_ctr.write(48, 2)
base_gpio_ctr.write(60, 2)
base_gpio_ctr.write(72, 2)
base_gpio_ctr.write(84, 2)

# (3)
# sets the desired coarse delay channel by channel
# to change the coarse delay
# about 250 us -> 50km of fiber (to be defined)

# Examples of coarse delays
# Note: The `delay_bob` and `delay_alice` variables must be defined before use.
# For example:
# delay_bob = 10
# delay_alice = 19
#
# These values are found experimentally to obtain preliminary temporal alignment between the nodes.
# Coarse delays for Bob's modulators
base_gpio_ctr.write(df_b_IM1 * 4, 0 + delay_bob) # Applies the coarse delay to Bob's IM1 modulator
base_gpio_ctr.write(df_b_IM2 * 4, 5 + delay_bob) # Applies the coarse delay to Bob's IM2 modulator
# Phase
base_gpio_ctr.write(df_b_PM * 4, 7 + delay_bob) # Applies the coarse delay to Bob's PM modulator

# Coarse delays for Alice's modulators
base_gpio_ctr.write(df_a_IM1 * 4, 0 + delay_alice) # Applies the coarse delay to Alice's IM1 modulator
base_gpio_ctr.write(df_a_IM2 * 4, 5 + delay_alice) # Applies the coarse delay to Alice's IM2 modulator
# Phase
base_gpio_ctr.write(df_a_PM * 4, 7 + delay_alice) # Applies the coarse delay to Alice's PM modulator


# The following code demonstrates writing specific coarse delay values directly.
# Coarse delays
# 1 intensity 1 Alice
# 4 intensity
base_gpio_ctr.write(1 * 4, 61882)
base_gpio_ctr.write(4 * 4, 61882)
base_gpio_ctr.write(7 * 4, 61882)
base_gpio_sma.write(0, 0x2)

# Formula to understand the coarse delay
# (250e-6)/4e-9
(250e-6-2.4e-6-260e-9+220e-9 -30e-9)/4e-9

# Set all delays to zero
# This is a good practice to reset the system for a new experiment.
base_gpio_ctr.write(1 * 4, 0)
base_gpio_ctr.write(4 * 4, 0)
base_gpio_ctr.write(7 * 4, 0)
base_gpio_ctr.write(10 * 4, 0)
base_gpio_ctr.write(13 * 4, 0)
base_gpio_ctr.write(17 * 4, 0)
base_gpio_ctr.write(20 * 4, 0)
base_gpio_ctr.write(23 * 4, 0)
