#!/bin/bash

# Requested parameter
REMOTE_USER="root"               # Remote user
REMOTE_IP=$1                     # IP of remote server
REMOTE_PASSWORD=$2               # Password of remote server
LOCAL_DESTINATION_PATH=$3        # Local directory where to move log
INTERFACE=$4			 # Interface to run suricata
THRESHOLD=$5                     # Protocol's threshold 
TIME=$6				 # Time to wait before stopping suricata
MAKEFILE_PATH=$7		 # Path of Makefile of Suricata's Openwrt package

CONFIG_FLAGS_FILE="/tmp/config_flags.txt" # Where to save python result
REMOTE_EVE_FILE="/var/log/suricata/eve.json" # Where to search for Suricata log
REMOTE_TMP="/tmp"

# Check if all parameters have been passed
if [[ -z "$REMOTE_IP" || -z "$REMOTE_PASSWORD" || -z "$LOCAL_DESTINATION_PATH" || -z "$INTERFACE" || -z "$THRESHOLD" || -z "$TIME" || -z "$MAKEFILE_PATH" ]]; then
    echo "Usage: $0 <REMOTE_IP> <REMOTE_PASSWORD> <LOCAL_DESTINATION_PATH> <SURICATA_INTERFACE> <PROTOCOL_THRESHOLD> <SLEEP_TIME> <MAKEFILE_PATH>"
    echo 'Example: ./suricata-configure.sh "192.168.56.2" "admin" "/home/kali/suricata-log" "eth0" 1 10 /home/kali/openwrt/feeds/packages/net/suricata'
    exit 1
fi

# Suricata stop in case it's running
echo "Stop Suricata on $REMOTE_IP if running..."
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "ash /sbin/service suricata stop"

echo "Removing old eve.json if present"
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "rm $REMOTE_EVE_FILE"

# Suricata start
echo "Init Suricata on $REMOTE_IP..."
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "suricata -c /etc/suricata/suricata.yaml -i $INTERFACE -D"

# Sleep of passed time
echo "Sleep..."
sleep $TIME

# Suricata stop
echo "Stop Suricata on $REMOTE_IP..."
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "ash /sbin/service suricata stop"

rm $LOCAL_DESTINATION_PATH/eve.json
# Transfer of file eve.json
echo "Transfer eve.json from remote host to local host..."
sshpass -p "$REMOTE_PASSWORD" scp -O $REMOTE_USER@$REMOTE_IP:$REMOTE_EVE_FILE $LOCAL_DESTINATION_PATH

# Check file transfer's status
if [[ $? -eq 0 ]]; then
    echo "Transfer successfully completed. File saved in: $LOCAL_DESTINATION_PATH"
else
    echo "Error during file's transfer."
    exit 1
fi

# Parsing of log file
echo "Parsing of log file"
python suricata-log-parser.py $LOCAL_DESTINATION_PATH/eve.json $THRESHOLD > "$CONFIG_FLAGS_FILE"

# Checking if the python script ended well
if [[ $? -ne 0 ]]; then
    echo "Error during python script run."
    exit 1
fi

# Definition of flags' variable
CONFIG_FLAGS=$(cat "$CONFIG_FLAGS_FILE")
echo "Generated flags:"
echo "$CONFIG_FLAGS"

MAKEFILE="$MAKEFILE_PATH/Makefile_default"
NEW_MAKEFILE="$MAKEFILE_PATH/Makefile"
if [[ -f "$MAKEFILE" ]]; then
    echo "Append generated flags to Suricata's Makefile..."
    
    cp "$MAKEFILE" "$NEW_MAKEFILE"

    # Append flags    
    sed -i "/^CONFIGURE_ARGS += \\\\/a \ \ $CONFIG_FLAGS \\\\" "$NEW_MAKEFILE"
    echo "Flag added!"
else
    echo "Makefile not found: $MAKEFILE"
    exit 1
fi

echo "Compiling Suricata package"
cd openwrt
make package/suricata/clean V=sc -j$(nproc) 

make package/suricata/compile V=sc -j$(nproc) &
MAKE_PID=$!

# Wait for make process to finish
wait $MAKE_PID

if [[ $? -ne 0 ]]; then
    echo "Error compiling suricata package."
    exit 1
fi

# Find the suricata compiled package
IPK_FILE=$(find bin/packages -name "suricata*.ipk" | head -n 1)

if [[ -z "$IPKG_FILE" ]]; then
    echo "Error: Suricata package not found."
    exit 1
fi

# Moving Suricata package to remote host
echo "Moving Suricata package to remote host"
sshpass -p "$REMOTE_PASSWORD" scp -O $IPK_FILE $REMOTE_USER@$REMOTE_IP:$REMOTE_TMP

if [[ $? -eq 0 ]]; then
    echo "Suricata package transferred successfully!"
else
    echo "Error during transfer of Suricata package."
    exit 1
fi

# Removing old Suricata
echo "Removing old Suricata package"
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "opkg remove suricata --force-depends"

# Installing Suricata
echo "Installing Suricata on remote host"
sshpass -p "$REMOTE_PASSWORD" ssh $REMOTE_USER@$REMOTE_IP "opkg install $REMOTE_TMP/suricata*.ipk"
