from pydantic import Field
from pydantic_settings import BaseSettings, SettingsConfigDict


class Settings(BaseSettings):
    """Shared configuration for all services."""

    # --- Infrastructure ---
    DATABASE_URL: str = Field(
        "postgresql+psycopg2://shm:shm@postgres:5432/SHM_ProjectDB",
        env="DATABASE_URL",
    )

    INFLUX_URL: str = Field("http://influxdb:8086", env="INFLUX_URL")
    INFLUX_ORG: str = Field("shm", env="INFLUX_ORG")
    INFLUX_TOKEN: str = Field("super-secret-token", env="INFLUX_TOKEN")

    RAW_BUCKET: str = Field("Raw_Data", env="RAW_BUCKET")
    CLEANED_BUCKET: str = Field("Cleaned_Data", env="CLEANED_BUCKET")
    ANALYZED_BUCKET: str = Field("Analyzed_Data", env="ANALYZED_BUCKET")
    OMA_BUCKET: str = Field("OMA_Data", env="OMA_BUCKET")

    # --- Service URLs ---
    METADATA_API_URL: str = Field("http://metadata_api:8001", env="METADATA_API_URL")
    CLEANING_API_URL: str = Field("http://cleaning_api:8030", env="CLEANING_API_URL")

    # --- Event bus (Redis Streams) ---
    REDIS_URL: str = Field("redis://redis:6379/0", env="REDIS_URL")
    STREAM_RAW_INGESTED: str = Field("shm_raw_ingested", env="STREAM_RAW_INGESTED")
    STREAM_CLEANED_READY: str = Field("shm_cleaned_ready", env="STREAM_CLEANED_READY")
    STREAM_ANALYSIS_DONE: str = Field("shm_analysis_done", env="STREAM_ANALYSIS_DONE")
    GROUP_CLEANING: str = Field("cleaning_group", env="GROUP_CLEANING")
    GROUP_ANALYSIS: str = Field("analysis_group", env="GROUP_ANALYSIS")

    # --- File-system I/O ---
    CLEANING_INPUT_DIR: str = Field("/data/incoming", env="CLEANING_INPUT_DIR")
    CLEANING_OUTPUT_DIR: str = Field("/data/cleaned", env="CLEANING_OUTPUT_DIR")

    # --- Ingestion / Watcher ---
    HOT_FOLDER: str = Field("/data/incoming", env="HOT_FOLDER")
    SIZE_MIN_MB: int = Field(10, env="SIZE_MIN_MB")
    STABLE_SECONDS: int = Field(60, env="STABLE_SECONDS")
    INGEST_URL: str = Field("http://ingestion_api:8010/ingest", env="INGEST_URL")

    MONITORED_STRUCTURE: str = Field("Sanctuary_Vicoforte", env="MONITORED_STRUCTURE")

    # --- Cleaning defaults ---
    CLEANING_HP: float = Field(0.5, env="CLEANING_HP")
    CLEANING_LP: float = Field(15.0, env="CLEANING_LP")
    CLEANING_DOWNSAMPLE: int = Field(2, env="CLEANING_DOWNSAMPLE")
    CLEANING_FILTER_ORDER: int = Field(4, env="CLEANING_FILTER_ORDER")
    CLEANING_DETREND_ORDER: int = Field(1, env="CLEANING_DETREND_ORDER")

    # --- Analysis defaults ---
    ANALYSIS_FREQ_MIN: float = Field(0.5, env="ANALYSIS_FREQ_MIN")
    ANALYSIS_FREQ_MAX: float = Field(20.0, env="ANALYSIS_FREQ_MAX")
    ANALYSIS_MAX_MODES: int = Field(8, env="ANALYSIS_MAX_MODES")
    ANALYSIS_MAC_THRESHOLD: float = Field(0.90, env="ANALYSIS_MAC_THRESHOLD")
    ANALYSIS_FREQ_REL_TOL: float = Field(0.05, env="ANALYSIS_FREQ_REL_TOL")
    ANALYSIS_CLUSTER_TOL: float = Field(0.01, env="ANALYSIS_CLUSTER_TOL")

    # --- Authentication ---
    UI_USERNAME: str = Field("admin", env="UI_USERNAME")
    UI_PASSWORD: str = Field("changeme", env="UI_PASSWORD")

    # Load .env locally; ignore unrelated env vars safely
    model_config = SettingsConfigDict(
        env_file=".env",
        env_file_encoding="utf-8",
        case_sensitive=True,
        extra="ignore",
    )


settings = Settings()
