from __future__ import annotations

from typing import Any, Dict, Optional

import httpx

from .settings import settings
from .logging import setup_logging

logger = setup_logging("shm.metadata_client")


def _base() -> str:
    return (settings.METADATA_API_URL or "http://metadata_api:8001").rstrip("/")


def _request(method: str, path: str, json: Optional[dict] = None, timeout: float = 10.0) -> Any:
    url = f"{_base()}{path}"
    with httpx.Client(timeout=timeout) as client:
        resp = client.request(method, url, json=json)
        if resp.status_code >= 400:
            raise RuntimeError(f"Metadata API {method} {path} failed: {resp.status_code} {resp.text}")
        if resp.text:
            return resp.json()
        return None


# --- Structures ---

def get_or_create_structure_id(name: str) -> int:
    """Resolve structure_id by name; create if missing."""
    # Try by-name (added in microservice refactor)
    try:
        data = _request("GET", f"/structures/by-name/{name}")
        return int(data["structure_id"])
    except Exception:
        # Fallback: list and search
        try:
            items = _request("GET", "/structures")
            for it in items:
                if it.get("name") == name:
                    return int(it["structure_id"])
        except Exception:
            pass
        created = _request("POST", "/structures/", json={"name": name, "location": ""})
        return int(created["structure_id"])


# --- Uploads ---

def create_upload(payload: Dict[str, Any]) -> int:
    data = _request("POST", "/uploads/", json=payload)
    return int(data["upload_id"])


def mark_upload_success(upload_id: int, points: int) -> None:
    _request("POST", f"/uploads/{upload_id}/success", json={"points": points})


def mark_upload_failed(upload_id: int, error_message: str) -> None:
    _request("POST", f"/uploads/{upload_id}/failed", json={"error_message": error_message})


def get_upload(upload_id: int) -> Dict[str, Any]:
    return _request("GET", f"/uploads/{upload_id}")


# --- Cleaning runs (Influx-based) ---

def create_cleaning_run(payload: Dict[str, Any]) -> int:
    data = _request("POST", "/cleaning/runs", json=payload)
    return int(data["cleaning_id"])


def complete_cleaning_run(cleaning_id: int, points_read: int, points_written: int) -> None:
    """Mark an Influx-based cleaning run as success."""
    _request(
        "POST",
        f"/cleaning/runs/{cleaning_id}/success",
        json={"points_read": points_read, "points_written": points_written},
    )


def fail_cleaning_run(cleaning_id: int, error_message: str) -> None:
    _request("POST", f"/cleaning/runs/{cleaning_id}/failed", json={"error_message": error_message})



# --- Config ---

def get_config(service: str) -> Dict[str, Any]:
    return _request("GET", f"/config/{service}")


# --- Analysis runs ---

def create_analysis_run(payload: Dict[str, Any]) -> int:
    data = _request("POST", "/analysis-runs/", json=payload)
    return int(data["run_id"])


def complete_analysis_run(run_id: int, notes: str | None = None) -> None:
    _request("POST", f"/analysis-runs/{run_id}/success", json={"notes": notes})


def fail_analysis_run(run_id: int, error_message: str) -> None:
    _request("POST", f"/analysis-runs/{run_id}/failed", json={"error_message": error_message})


# --- Sensors ---

def upsert_sensor(structure_id: int, name: str, sensor_type: str | None = None, unit: str | None = None, channel: str | None = None) -> Dict[str, Any]:
    payload = {"structure_id": structure_id, "name": name, "sensor_type": sensor_type, "unit": unit, "channel": channel}
    return _request("POST", "/sensors/upsert", json=payload)

def list_sensors(structure_id: int | None = None) -> list[Dict[str, Any]]:
    items = _request("GET", "/sensors") or []
    if structure_id is None:
        return items
    return [it for it in items if int(it.get("structure_id", -1)) == int(structure_id)]

def ensure_sensors(structure_id: int, sensor_names: list[str]) -> None:
    for nm in sensor_names:
        try:
            upsert_sensor(structure_id, nm)
        except Exception:
            # Best-effort: if upsert is not available or temporary error, ignore.
            continue

# --- System config helpers ---
def get_monitored_structure_name(default: str) -> str:
    try:
        cfg = get_config("system") or {}
        name = cfg.get("monitored_structure") or cfg.get("MONITORED_STRUCTURE")
        if name:
            return str(name)
    except Exception:
        pass
    return default


# --- OMA / Modal Identification ---

def list_oma_modes(structure_id: int | None = None) -> list[Dict[str, Any]]:
    if structure_id is None:
        return _request("GET", "/oma/modes") or []
    return _request("GET", f"/oma/modes?structure_id={structure_id}") or []

def create_oma_mode(payload: Dict[str, Any]) -> Dict[str, Any]:
    return _request("POST", "/oma/modes", json=payload)

def mark_oma_mode_seen(mode_id: int, payload: Dict[str, Any]) -> Dict[str, Any]:
    return _request("POST", f"/oma/modes/{mode_id}/seen", json=payload)

def create_oma_observation(payload: Dict[str, Any]) -> Dict[str, Any]:
    return _request("POST", "/oma/observations", json=payload)
