from typing import Iterable, Iterator, Sequence

from influxdb_client import InfluxDBClient, Point
from influxdb_client.client.write_api import SYNCHRONOUS

from .settings import settings


def get_influx_client() -> InfluxDBClient:
    return InfluxDBClient(url=settings.INFLUX_URL, token=settings.INFLUX_TOKEN, org=settings.INFLUX_ORG)


def ensure_buckets() -> None:
    """Ensure required InfluxDB buckets exist (idempotent)."""
    bucket_names = [
        settings.RAW_BUCKET,
        settings.CLEANED_BUCKET,
        settings.OMA_BUCKET,
        settings.ANALYZED_BUCKET,
    ]

    client = get_influx_client()
    try:
        orgs_api = client.organizations_api()
        orgs = orgs_api.find_organizations(org=settings.INFLUX_ORG)
        if not orgs:
            raise RuntimeError(f"InfluxDB org not found: {settings.INFLUX_ORG}")
        org_id = orgs[0].id

        buckets_api = client.buckets_api()
        for name in bucket_names:
            existing = buckets_api.find_bucket_by_name(name)
            if existing is None:
                buckets_api.create_bucket(bucket_name=name, org_id=org_id)
    finally:
        client.close()


def _batched(iterable: Iterable[Point], batch_size: int) -> Iterator[Sequence[Point]]:
    batch: list[Point] = []
    for item in iterable:
        batch.append(item)
        if len(batch) >= batch_size:
            yield batch
            batch = []
    if batch:
        yield batch


def write_points(points: Iterable[Point], bucket: str, chunk_size: int = 50_000) -> int:
    """Write points in chunks, keeping memory usage low. Returns total points written."""
    total = 0
    client = get_influx_client()
    write_api = client.write_api(write_options=SYNCHRONOUS)
    try:
        for chunk in _batched(points, chunk_size):
            write_api.write(bucket=bucket, org=settings.INFLUX_ORG, record=chunk)
            total += len(chunk)
        return total
    finally:
        write_api.close()
        client.close()
