from __future__ import annotations

from datetime import datetime
from typing import Optional

from pydantic import BaseModel, ConfigDict, Field


# -----------------------------
# Structures
# -----------------------------
class StructureCreate(BaseModel):
    name: str
    type: Optional[str] = None
    location: Optional[str] = None
    latitude: Optional[float] = None
    longitude: Optional[float] = None
    description: Optional[str] = None


class StructureRead(BaseModel):
    """Read schema aligned with ORM column names."""
    structure_id: int
    name: str
    location: Optional[str] = None
    structure_type: Optional[str] = None
    latitude: Optional[float] = None
    longitude: Optional[float] = None
    notes: Optional[str] = None

    model_config = ConfigDict(from_attributes=True)


# -----------------------------
# Sensors
# -----------------------------
class SensorCreate(BaseModel):
    structure_id: int
    name: str
    sensor_type: Optional[str] = None
    unit: Optional[str] = None
    channel: Optional[int] = None  # numeric channel index if applicable


class SensorRead(SensorCreate):
    sensor_id: int
    model_config = ConfigDict(from_attributes=True)


# -----------------------------
# Uploads
# -----------------------------
class UploadCreate(BaseModel):
    structure_id: int
    filename: str

    # These may be unknown at creation time, so keep optional.
    file_path: Optional[str] = None
    file_size_mb: Optional[float] = None
    sensor_count: Optional[int] = None

    # Client-facing names; ORM stores as file_start_time/file_end_time
    start_time: Optional[datetime] = None
    end_time: Optional[datetime] = None

    status: str = "pending"
    notes: Optional[str] = None


class UploadRead(BaseModel):
    upload_id: int
    structure_id: int
    filename: str

    file_path: Optional[str] = None
    file_size_mb: Optional[float] = None
    sensor_count: Optional[int] = None

    file_start_time: Optional[datetime] = None
    file_end_time: Optional[datetime] = None

    status: str
    points_written: Optional[int] = None
    error_message: Optional[str] = None
    notes: Optional[str] = None

    upload_timestamp: datetime
    started_at: datetime
    finished_at: Optional[datetime] = None

    model_config = ConfigDict(from_attributes=True)


# -----------------------------
# Analysis runs
# -----------------------------
class AnalysisRunCreate(BaseModel):
    structure_id: int
    upload_id: int
    status: str = "running"
    notes: Optional[str] = None


class AnalysisRunRead(BaseModel):
    run_id: int
    structure_id: int
    upload_id: int
    status: str
    notes: Optional[str] = None
    started_at: datetime
    finished_at: Optional[datetime] = None

    model_config = ConfigDict(from_attributes=True)


# -----------------------------
# Cleaning runs
# -----------------------------
class CleaningRunCreate(BaseModel):
    upload_id: int
    structure_id: int

    original_fs: Optional[float] = None
    cleaned_fs: Optional[float] = None

    hp_cutoff: Optional[float] = None
    lp_cutoff: Optional[float] = None
    downsample_factor: Optional[int] = None
    filter_order: Optional[int] = None
    detrend_order: Optional[int] = None

    points_read: Optional[int] = None
    points_written: Optional[int] = None

    status: str = "running"
    error_message: Optional[str] = None
    notes: Optional[str] = None


class CleaningRunRead(BaseModel):
    cleaning_id: int
    upload_id: int
    structure_id: int

    started_at: datetime
    finished_at: Optional[datetime] = None

    original_fs: Optional[float] = None
    cleaned_fs: Optional[float] = None

    hp_cutoff: Optional[float] = None
    lp_cutoff: Optional[float] = None
    downsample_factor: Optional[int] = None
    filter_order: Optional[int] = None
    detrend_order: Optional[int] = None

    points_read: Optional[int] = None
    points_written: Optional[int] = None

    status: str
    error_message: Optional[str] = None
    notes: Optional[str] = None

    model_config = ConfigDict(from_attributes=True)


# -----------------------------
# OMA / Modal Identification
# -----------------------------
class OMAModeCreate(BaseModel):
    structure_id: int
    name: Optional[str] = None  # Can be None initially; updated via mark_oma_mode_seen
    ref_frequency_hz: float
    ref_shape_json: str  # JSON string


class OMAModeRead(BaseModel):
    mode_id: int
    structure_id: int
    name: Optional[str] = None
    ref_frequency_hz: Optional[float] = None
    ref_shape_json: Optional[str] = None

    created_at: datetime
    updated_at: datetime
    last_seen_at: datetime
    active: bool

    model_config = ConfigDict(from_attributes=True)


class OMAObservationCreate(BaseModel):
    run_id: int
    mode_id: int
    frequency_hz: float
    damping_ratio: Optional[float] = None
    mac: Optional[float] = None
    quality: Optional[float] = None


class OMAObservationRead(BaseModel):
    observation_id: int
    run_id: int
    mode_id: int
    timestamp: datetime
    frequency_hz: float
    damping_ratio: Optional[float] = None
    mac: Optional[float] = None
    quality: Optional[float] = None

    model_config = ConfigDict(from_attributes=True)
