from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from pydantic import BaseModel
from ..database import get_db
from .. import schemas, crud

router = APIRouter(prefix="/uploads", tags=["uploads"])


@router.get("/", response_model=list[schemas.UploadRead])
def list_uploads(skip: int = 0, limit: int = 100, db: Session = Depends(get_db)):
    return crud.list_uploads(db, skip=skip, limit=limit)


@router.get("/{upload_id}", response_model=schemas.UploadRead)
def get_upload(upload_id: int, db: Session = Depends(get_db)):
    obj = crud.get_upload(db, upload_id)
    if not obj:
        raise HTTPException(status_code=404, detail="Upload not found")
    return obj

@router.post("/", response_model=schemas.UploadRead)
def create_upload(payload: schemas.UploadCreate, db: Session = Depends(get_db)):
    return crud.create_upload(db, payload)

class UploadSuccessBody(BaseModel):
    points: int

class UploadFailBody(BaseModel):
    error_message: str

@router.post("/{upload_id}/success", response_model=schemas.UploadRead)
def mark_success(upload_id: int, body: UploadSuccessBody, db: Session = Depends(get_db)):
    updated = crud.mark_upload_success(db, upload_id, body.points)
    if not updated:
        raise HTTPException(status_code=404, detail="Upload not found")
    return updated

@router.post("/{upload_id}/failed", response_model=schemas.UploadRead)
def mark_failed(upload_id: int, body: UploadFailBody, db: Session = Depends(get_db)):
    updated = crud.mark_upload_failed(db, upload_id, body.error_message)
    if not updated:
        raise HTTPException(status_code=404, detail="Upload not found")
    return updated
