from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import Session

from .. import crud, schemas
from ..database import get_db

from shm_shared.settings import settings

router = APIRouter(prefix="/structures", tags=["structures"])


@router.get("/by-name/{name}", response_model=schemas.StructureRead)
def get_structure_by_name(name: str, db: Session = Depends(get_db)):
    """Get a structure by its unique name."""
    obj = crud.get_structure_by_name(db, name)
    if not obj:
        raise HTTPException(status_code=404, detail="Structure not found")
    return obj


@router.post("/", response_model=schemas.StructureRead)
def create_structure(payload: schemas.StructureCreate, db: Session = Depends(get_db)):
    return crud.create_structure(db, payload)


@router.get("/", response_model=list[schemas.StructureRead])
def list_structures(db: Session = Depends(get_db)):
    return crud.list_structures(db)


@router.delete("/{structure_id}", status_code=204)
def delete_structure(structure_id: int, db: Session = Depends(get_db)):
    obj = crud.get_structure(db, structure_id)
    if not obj:
        raise HTTPException(status_code=404, detail="Structure not found")

    if obj.name == settings.MONITORED_STRUCTURE:
        raise HTTPException(status_code=403, detail="Cannot delete the monitored/default structure.")

    try:
        ok = crud.delete_structure(db, structure_id)
    except IntegrityError:
        raise HTTPException(status_code=409, detail="Cannot delete structure: it has related rows (uploads/runs/sensors).")
    if not ok:
        raise HTTPException(status_code=404, detail="Structure not found")
    return None


@router.delete("/by-name/{name}", status_code=204)
def delete_structure_by_name(name: str, db: Session = Depends(get_db)):
    if name == settings.MONITORED_STRUCTURE:
        raise HTTPException(status_code=403, detail="Cannot delete the monitored/default structure.")

    try:
        ok = crud.delete_structure_by_name(db, name)
    except IntegrityError:
        raise HTTPException(status_code=409, detail="Cannot delete structure: it has related rows (uploads/runs/sensors).")
    if not ok:
        raise HTTPException(status_code=404, detail="Structure not found")
    return None
