from fastapi import APIRouter, Depends
from sqlalchemy.orm import Session

from ..database import get_db
from .. import schemas, crud

router = APIRouter(prefix="/oma", tags=["oma"])

@router.get("/modes", response_model=list[schemas.OMAModeRead])
def list_modes(structure_id: int | None = None, db: Session = Depends(get_db)):
    return crud.list_oma_modes(db, structure_id=structure_id)

@router.post("/modes", response_model=schemas.OMAModeRead)
def create_mode(payload: schemas.OMAModeCreate, db: Session = Depends(get_db)):
    return crud.create_oma_mode(db, payload)

@router.post("/modes/{mode_id}/seen", response_model=schemas.OMAModeRead)
def mark_seen(mode_id: int, payload: dict, db: Session = Depends(get_db)):
    return crud.update_oma_mode_seen(
        db,
        mode_id,
        ref_frequency_hz=payload.get("ref_frequency_hz"),
        ref_shape_json=payload.get("ref_shape_json"),
    )

@router.get("/observations", response_model=list[schemas.OMAObservationRead])
def list_observations(structure_id: int | None = None, limit: int = 200, db: Session = Depends(get_db)):
    return crud.list_oma_observations(db, structure_id=structure_id, limit=limit)

@router.post("/observations", response_model=schemas.OMAObservationRead)
def create_observation(payload: schemas.OMAObservationCreate, db: Session = Depends(get_db)):
    return crud.create_oma_observation(db, payload)
