from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session

from ..database import get_db
from .. import crud

router = APIRouter(prefix="/config", tags=["config"])


def _rows_to_dict(rows) -> dict:
    # rows are models.PipelineConfig objects
    return {r.key: r.value for r in rows}


@router.get("/{service}")
def get_service_config(service: str, db: Session = Depends(get_db)):
    rows = crud.get_pipeline_config(db, service=service)
    return _rows_to_dict(rows)


@router.put("/{service}")
def update_service_config(service: str, payload: dict, db: Session = Depends(get_db)):
    if payload is None:
        payload = {}

    if not isinstance(payload, dict):
        raise HTTPException(status_code=422, detail="Payload must be a JSON object/dict")

    for k, v in payload.items():
        crud.upsert_pipeline_config(
            db,
            service=service,
            key=str(k),
            value=None if v is None else str(v),
        )

    rows = crud.get_pipeline_config(db, service=service)
    return _rows_to_dict(rows)
