"""
API endpoints for managing cleaning runs.
"""
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from typing import List
from pydantic import BaseModel

from .. import crud, schemas
from ..database import get_db

router = APIRouter(prefix="/cleaning", tags=["cleaning"])


@router.post("/runs", response_model=schemas.CleaningRunRead)
def create_cleaning_run(payload: schemas.CleaningRunCreate, db: Session = Depends(get_db)):
    """Create a cleaning run record (status defaults to 'running')."""
    return crud.create_cleaning_run(db, payload)


class CleaningSuccessBody(BaseModel):
    points_read: int
    points_written: int


class CleaningFailedBody(BaseModel):
    error_message: str


@router.post("/runs/{cleaning_id}/success", response_model=schemas.CleaningRunRead)
def mark_cleaning_success(cleaning_id: int, body: CleaningSuccessBody, db: Session = Depends(get_db)):
    obj = crud.mark_cleaning_success(db, cleaning_id, body.points_read, body.points_written)
    if not obj:
        raise HTTPException(status_code=404, detail="Cleaning run not found")
    return obj


@router.post("/runs/{cleaning_id}/failed", response_model=schemas.CleaningRunRead)
def mark_cleaning_failed(cleaning_id: int, body: CleaningFailedBody, db: Session = Depends(get_db)):
    obj = crud.mark_cleaning_failed(db, cleaning_id, body.error_message)
    if not obj:
        raise HTTPException(status_code=404, detail="Cleaning run not found")
    return obj


@router.get("/runs", response_model=List[schemas.CleaningRunRead])
def list_cleaning_runs(
    skip: int = 0,
    limit: int = 100,
    db: Session = Depends(get_db),
):
    """List all cleaning runs."""
    return crud.list_cleaning_runs(db, skip=skip, limit=limit)


@router.get("/runs/{cleaning_id}", response_model=schemas.CleaningRunRead)
def get_cleaning_run(cleaning_id: int, db: Session = Depends(get_db)):
    """Get a specific cleaning run by ID."""
    run = crud.get_cleaning_run(db, cleaning_id)
    if not run:
        raise HTTPException(status_code=404, detail="Cleaning run not found")
    return run


@router.get("/runs/upload/{upload_id}", response_model=schemas.CleaningRunRead)
def get_cleaning_run_by_upload(upload_id: int, db: Session = Depends(get_db)):
    """Get the cleaning run for a specific upload."""
    run = crud.get_cleaning_run_by_upload(db, upload_id)
    if not run:
        raise HTTPException(status_code=404, detail="No cleaning run found for this upload")
    return run
