from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session
from .. import crud, database, schemas
from pydantic import BaseModel

router = APIRouter(prefix="/analysis-runs", tags=["analysis-runs"])


@router.post("/", response_model=schemas.AnalysisRunRead, status_code=201)
def create_analysis_run(payload: schemas.AnalysisRunCreate, db: Session = Depends(database.get_db)):
    return crud.create_analysis_run(db, payload)


class AnalysisSuccessBody(BaseModel):
    notes: str | None = None


class AnalysisFailedBody(BaseModel):
    error_message: str


@router.post("/{run_id}/success", response_model=schemas.AnalysisRunRead)
def mark_success(run_id: int, body: AnalysisSuccessBody, db: Session = Depends(database.get_db)):
    obj = crud.mark_analysis_success(db, run_id, notes=body.notes)
    if not obj:
        raise HTTPException(status_code=404, detail="AnalysisRun not found")
    return obj


@router.post("/{run_id}/failed", response_model=schemas.AnalysisRunRead)
def mark_failed(run_id: int, body: AnalysisFailedBody, db: Session = Depends(database.get_db)):
    obj = crud.mark_analysis_failed(db, run_id, error_message=body.error_message)
    if not obj:
        raise HTTPException(status_code=404, detail="AnalysisRun not found")
    return obj

@router.get("/", response_model=list[schemas.AnalysisRunRead])
def list_analysis_runs(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    db: Session = Depends(database.get_db),
):
    return crud.list_analysis_runs(db, skip=skip, limit=limit)

@router.get("/{run_id}", response_model=schemas.AnalysisRunRead)
def get_analysis_run(run_id: int, db: Session = Depends(database.get_db)):
    run = crud.get_analysis_run(db, run_id)
    if not run:
        raise HTTPException(status_code=404, detail="AnalysisRun not found")
    return run

 
