from __future__ import annotations

from datetime import datetime
from sqlalchemy import (
    Column,
    Integer,
    String,
    DateTime,
    Float,
    Text,
    Boolean,
    ForeignKey,
    UniqueConstraint,
)
from sqlalchemy.orm import relationship

from .database import Base


# -----------------------------
# Structures & Sensors
# -----------------------------
class Structure(Base):
    __tablename__ = "structures"

    structure_id = Column(Integer, primary_key=True, index=True, autoincrement=True)
    name = Column(String(100), unique=True, nullable=False, index=True)
    # Optional descriptive fields
    location = Column(String(200), nullable=True)
    structure_type = Column(String(50), nullable=True)
    latitude = Column(Float, nullable=True)
    longitude = Column(Float, nullable=True)
    notes = Column(Text, nullable=True)

    # Children with single parent => delete-orphan is correct
    sensors = relationship(
        "Sensor",
        back_populates="structure",
        cascade="all, delete-orphan",
        passive_deletes=True,
        single_parent=True,
    )

    uploads = relationship(
        "UploadLog",
        back_populates="structure",
        cascade="all, delete-orphan",
        passive_deletes=True,
        single_parent=True,
        order_by="UploadLog.upload_id.desc()",
    )

    # RUNS are also linked to UploadLog, which we treat as the OWNER.
    # So this is a normal relationship (no delete-orphan here).
    runs = relationship(
        "AnalysisRun",
        back_populates="structure",
        passive_deletes=True,
        order_by="AnalysisRun.run_id.desc()",
    )


class Sensor(Base):
    __tablename__ = "sensors"
    __table_args__ = (
        # A sensor name must be unique per structure
        UniqueConstraint("structure_id", "name", name="uq_sensor_per_structure"),
    )

    sensor_id = Column(Integer, primary_key=True, autoincrement=True)
    structure_id = Column(
        Integer,
        ForeignKey("structures.structure_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )
    name = Column(String(100), nullable=False, index=True)
    sensor_type = Column(String(50), nullable=True)  # e.g., "accelerometer"
    unit = Column(String(32), nullable=True)         # e.g., "g", "m/s^2"
    channel = Column(String(50), nullable=True)      # optional label

    structure = relationship("Structure", back_populates="sensors")


# -----------------------------
# Uploads & Analysis Runs
# -----------------------------
class UploadLog(Base):
    __tablename__ = "uploads_log"
    __table_args__ = (
        UniqueConstraint("structure_id", "filename", "upload_timestamp", 
                        name="uq_structure_file_timestamp"),
    )

    upload_id = Column(Integer, primary_key=True, autoincrement=True)
    structure_id = Column(
        Integer,
        ForeignKey("structures.structure_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    filename = Column(String(300), nullable=False)
    file_path = Column(String(500), nullable=True)
    file_size_mb = Column(Float, nullable=True)
    sensor_count = Column(Integer, nullable=True)
    upload_timestamp = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    file_start_time = Column(DateTime(timezone=True), nullable=True)  # t0 from file (actual data start)
    file_end_time = Column(DateTime(timezone=True), nullable=True)    # computed from t0 + duration
    points_written = Column(Integer, nullable=True)

    started_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    finished_at = Column(DateTime(timezone=True), nullable=True)

    status = Column(String(32), nullable=False, default="pending")  # pending|success|failed
    cleaning_attempted = Column(Boolean, default=False, nullable=False)  # Track if cleaner has processed this
    analysis_attempted = Column(Boolean, default=False, nullable=False)  # Track if analyzer has processed this
    error_message = Column(String, nullable=True)
    notes = Column(String, nullable=True)

    structure = relationship("Structure", back_populates="uploads")

    # UploadLog is the OWNER of runs (each run belongs to one upload)
    runs = relationship(
        "AnalysisRun",
        back_populates="upload",
        cascade="all, delete-orphan",
        passive_deletes=True,
        single_parent=True,
        order_by="AnalysisRun.run_id.desc()",
    )

    # UploadLog is the OWNER of cleaning runs (each upload has one cleaning run)
    cleaning_runs = relationship(
        "CleaningRun",
        back_populates="upload",
        cascade="all, delete-orphan",
        passive_deletes=True,
        single_parent=True,
        order_by="CleaningRun.cleaning_id.desc()",
    )


class AnalysisRun(Base):
    __tablename__ = "analysis_runs"

    run_id = Column(Integer, primary_key=True, autoincrement=True)
    structure_id = Column(
        Integer,
        ForeignKey("structures.structure_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )
    upload_id = Column(
        Integer,
        ForeignKey("uploads_log.upload_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    started_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    finished_at = Column(DateTime(timezone=True), nullable=True)

    # running | success | failed
    status = Column(String(32), nullable=False, default="running")
    notes = Column(Text, nullable=True)

    structure = relationship("Structure", back_populates="runs")
    upload = relationship("UploadLog", back_populates="runs")



# -----------------------------
# Cleaning Runs
# -----------------------------
class CleaningRun(Base):
    __tablename__ = "cleaning_runs"
    __table_args__ = (
        UniqueConstraint("upload_id", name="uq_cleaning_per_upload"),
    )

    cleaning_id = Column(Integer, primary_key=True, autoincrement=True)
    upload_id = Column(
        Integer,
        ForeignKey("uploads_log.upload_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )
    structure_id = Column(
        Integer,
        ForeignKey("structures.structure_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    started_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    finished_at = Column(DateTime(timezone=True), nullable=True)

    # running | success | failed
    status = Column(String(32), nullable=False, default="running")

    # Cleaning parameters (for reproducibility)
    hp_cutoff = Column(Float, nullable=True)           # high-pass cutoff (Hz)
    lp_cutoff = Column(Float, nullable=True)           # low-pass cutoff (Hz)
    downsample_factor = Column(Integer, nullable=True) # downsampling factor
    filter_order = Column(Integer, nullable=True)      # Butterworth filter order
    detrend_order = Column(Integer, nullable=True)     # polynomial detrend order
    original_fs = Column(Float, nullable=True)         # original sampling frequency
    cleaned_fs = Column(Float, nullable=True)          # cleaned sampling frequency

    # Output tracking
    points_read = Column(Integer, nullable=True)       # points read from Raw_Data
    points_written = Column(Integer, nullable=True)    # points written to Cleaned_Data

    # Error handling
    error_message = Column(Text, nullable=True)
    notes = Column(Text, nullable=True)

    upload = relationship("UploadLog", back_populates="cleaning_runs")
    structure = relationship("Structure")



# -----------------------------
# Runtime Config (Decoupled UI)
# -----------------------------

class PipelineConfig(Base):
    """
    Key/value config store for runtime-tunable parameters.
    - service: 'ingestion' | 'cleaning' | 'analysis' (or other)
    """
    __tablename__ = "pipeline_config"
    __table_args__ = (
        UniqueConstraint("service", "key", name="uq_pipeline_config_service_key"),
    )

    config_id = Column(Integer, primary_key=True, autoincrement=True)
    service = Column(String(64), nullable=False, index=True)
    key = Column(String(128), nullable=False, index=True)
    value = Column(String(512), nullable=True)
    updated_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)


# -----------------------------
# OMA (Operational Modal Analysis)
# -----------------------------

class OMAMode(Base):
    __tablename__ = "oma_modes"

    mode_id = Column(Integer, primary_key=True, autoincrement=True)
    structure_id = Column(
        Integer,
        ForeignKey("structures.structure_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # Reference (tracked) properties
    name = Column(String(64), nullable=True)
    ref_frequency_hz = Column(Float, nullable=False)
    ref_shape_json = Column(Text, nullable=False)  # JSON list of floats

    created_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    updated_at = Column(DateTime(timezone=True), default=datetime.utcnow, onupdate=datetime.utcnow, nullable=False)
    last_seen_at = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False)
    active = Column(Boolean, default=True, nullable=False)

    structure = relationship("Structure")
    observations = relationship(
        "OMAObservation",
        back_populates="mode",
        cascade="all, delete-orphan",
        passive_deletes=True,
        single_parent=True,
        order_by="OMAObservation.observation_id.desc()",
    )


class OMAObservation(Base):
    __tablename__ = "oma_observations"

    observation_id = Column(Integer, primary_key=True, autoincrement=True)

    run_id = Column(
        Integer,
        ForeignKey("analysis_runs.run_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    mode_id = Column(
        Integer,
        ForeignKey("oma_modes.mode_id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    timestamp = Column(DateTime(timezone=True), default=datetime.utcnow, nullable=False, index=True)
    frequency_hz = Column(Float, nullable=False)
    damping_ratio = Column(Float, nullable=True)
    mac = Column(Float, nullable=True)
    quality = Column(Float, nullable=True)

    run = relationship("AnalysisRun")
    mode = relationship("OMAMode", back_populates="observations")
