from typing import List, Optional
from datetime import datetime

from sqlalchemy.orm import Session
from sqlalchemy.exc import IntegrityError
from sqlalchemy import func

from . import models, schemas


# -----------------------------
# Structures
# -----------------------------
def create_structure(db: Session, payload: schemas.StructureCreate) -> models.Structure:
    obj = models.Structure(
        name=payload.name,
        structure_type=payload.type,
        location=payload.location,
        latitude=payload.latitude,
        longitude=payload.longitude,
        notes=payload.description,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_structures(db: Session) -> List[models.Structure]:
    return db.query(models.Structure).all()

def get_structure(db: Session, structure_id: int) -> Optional[models.Structure]:
    return db.query(models.Structure).filter(models.Structure.structure_id == structure_id).first()

def get_structure_by_name(db: Session, name: str) -> Optional[models.Structure]:
    return db.query(models.Structure).filter(models.Structure.name == name).first()


def delete_structure(db: Session, structure_id: int) -> bool:
    obj = db.query(models.Structure).filter(models.Structure.structure_id == structure_id).first()
    if not obj:
        return False
    try:
        db.delete(obj)
        db.commit()
        return True
    except IntegrityError:
        db.rollback()
        raise


def delete_structure_by_name(db: Session, name: str) -> bool:
    obj = get_structure_by_name(db, name)
    if not obj:
        return False
    return delete_structure(db, obj.structure_id)


# -----------------------------
# Sensors
# -----------------------------
def create_sensor(db: Session, payload: schemas.SensorCreate) -> models.Sensor:
    obj = models.Sensor(
        structure_id=payload.structure_id,
        name=payload.name,
        sensor_type=payload.sensor_type,
        unit=payload.unit,
        channel=payload.channel,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def upsert_sensor(db: Session, payload: schemas.SensorCreate) -> models.Sensor:
    existing = (
        db.query(models.Sensor)
        .filter(models.Sensor.structure_id == payload.structure_id)
        .filter(models.Sensor.name == payload.name)
        .first()
    )
    if existing:
        existing.sensor_type = payload.sensor_type
        existing.unit = payload.unit
        existing.channel = payload.channel
        db.commit()
        db.refresh(existing)
        return existing
    return create_sensor(db, payload)


def list_sensors(db: Session, structure_id: Optional[int] = None) -> List[models.Sensor]:
    q = db.query(models.Sensor)
    if structure_id is not None:
        q = q.filter(models.Sensor.structure_id == structure_id)
    return q.all()


# -----------------------------
# Upload / Runs / Config (existing)
# -----------------------------
def create_upload(db: Session, payload: schemas.UploadCreate) -> models.UploadLog:
    obj = models.UploadLog(
        structure_id=payload.structure_id,
        filename=payload.filename,
        file_path=payload.file_path,
        file_size_mb=payload.file_size_mb,
        sensor_count=payload.sensor_count,
        file_start_time=payload.start_time,
        file_end_time=payload.end_time,
        status=payload.status,
        notes=payload.notes,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_uploads(db: Session, skip: int = 0, limit: int = 50) -> List[models.UploadLog]:
    return db.query(models.UploadLog).order_by(models.UploadLog.upload_id.desc()).offset(skip).limit(limit).all()


def get_upload(db: Session, upload_id: int) -> Optional[models.UploadLog]:
    return db.query(models.UploadLog).filter(models.UploadLog.upload_id == upload_id).first()


def mark_upload_success(db: Session, upload_id: int, points: int) -> Optional[models.UploadLog]:
    obj = get_upload(db, upload_id)
    if not obj:
        return None
    obj.status = "success"
    obj.points_written = points
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj


def mark_upload_failed(db: Session, upload_id: int, error_message: str) -> Optional[models.UploadLog]:
    obj = get_upload(db, upload_id)
    if not obj:
        return None
    obj.status = "failed"
    obj.error_message = error_message
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj


def create_cleaning_run(db: Session, payload: schemas.CleaningRunCreate) -> models.CleaningRun:
    obj = models.CleaningRun(
        upload_id=payload.upload_id,
        structure_id=payload.structure_id,
        original_fs=payload.original_fs,
        cleaned_fs=payload.cleaned_fs,
        hp_cutoff=payload.hp_cutoff,
        lp_cutoff=payload.lp_cutoff,
        downsample_factor=payload.downsample_factor,
        filter_order=payload.filter_order,
        detrend_order=payload.detrend_order,
        points_read=payload.points_read,
        points_written=payload.points_written,
        status=payload.status,
        error_message=payload.error_message,
        notes=payload.notes,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_cleaning_runs(db: Session, skip: int = 0, limit: int = 50) -> List[models.CleaningRun]:
    return db.query(models.CleaningRun).order_by(models.CleaningRun.cleaning_id.desc()).offset(skip).limit(limit).all()


def get_cleaning_run(db: Session, cleaning_id: int) -> Optional[models.CleaningRun]:
    return db.query(models.CleaningRun).filter(models.CleaningRun.cleaning_id == cleaning_id).first()


def get_cleaning_run_by_upload(db: Session, upload_id: int) -> Optional[models.CleaningRun]:
    return db.query(models.CleaningRun).filter(models.CleaningRun.upload_id == upload_id).first()


def mark_cleaning_success(db: Session, cleaning_id: int, points_read: int, points_written: int) -> Optional[models.CleaningRun]:
    obj = get_cleaning_run(db, cleaning_id)
    if not obj:
        return None
    obj.status = "success"
    obj.points_read = points_read
    obj.points_written = points_written
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj


def mark_cleaning_failed(db: Session, cleaning_id: int, error_message: str) -> Optional[models.CleaningRun]:
    obj = get_cleaning_run(db, cleaning_id)
    if not obj:
        return None
    obj.status = "failed"
    obj.error_message = error_message
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj


def create_analysis_run(db: Session, payload: schemas.AnalysisRunCreate) -> models.AnalysisRun:
    obj = models.AnalysisRun(
        upload_id=payload.upload_id,
        structure_id=payload.structure_id,
        status=payload.status,
        notes=payload.notes,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_analysis_runs(db: Session, skip: int = 0, limit: int = 50) -> List[models.AnalysisRun]:
    return db.query(models.AnalysisRun).order_by(models.AnalysisRun.run_id.desc()).offset(skip).limit(limit).all()


def get_analysis_run(db: Session, run_id: int) -> Optional[models.AnalysisRun]:
    return db.query(models.AnalysisRun).filter(models.AnalysisRun.run_id == run_id).first()


def mark_analysis_success(db: Session, run_id: int, notes: Optional[str] = None) -> Optional[models.AnalysisRun]:
    obj = get_analysis_run(db, run_id)
    if not obj:
        return None
    obj.status = "success"
    obj.notes = notes
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj


def mark_analysis_failed(db: Session, run_id: int, error_message: str) -> Optional[models.AnalysisRun]:
    obj = get_analysis_run(db, run_id)
    if not obj:
        return None
    obj.status = "failed"
    obj.notes = error_message
    obj.finished_at = func.now()
    db.commit()
    db.refresh(obj)
    return obj



# ----------------------------- 
# OMA Modes 
# ----------------------------- 
def create_oma_mode(db: Session, payload: schemas.OMAModeCreate) -> models.OMAMode:
    now = datetime.utcnow()
    obj = models.OMAMode(
        structure_id=payload.structure_id,
        name=payload.name,
        ref_frequency_hz=payload.ref_frequency_hz,
        ref_shape_json=payload.ref_shape_json,
        updated_at=now,
        last_seen_at=now,
        active=True,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_oma_modes(db: Session, structure_id: Optional[int] = None) -> List[models.OMAMode]:
    q = db.query(models.OMAMode)
    if structure_id is not None:
        q = q.filter(models.OMAMode.structure_id == structure_id)
    return q.all()


def get_oma_mode(db: Session, mode_id: int) -> Optional[models.OMAMode]:
    return db.query(models.OMAMode).filter(models.OMAMode.mode_id == mode_id).first()


def update_oma_mode_seen(
    db: Session,
    mode_id: int,
    ref_frequency_hz: Optional[float] = None,
    ref_shape_json: Optional[str] = None,
) -> Optional[models.OMAMode]:
    obj = get_oma_mode(db, mode_id)
    if not obj:
        return None

    if ref_frequency_hz is not None:
        obj.ref_frequency_hz = float(ref_frequency_hz)
    if ref_shape_json is not None:
        obj.ref_shape_json = str(ref_shape_json)

    now = datetime.utcnow()
    obj.last_seen_at = now
    obj.updated_at = now
    db.commit()
    db.refresh(obj)
    return obj


def create_oma_observation(db: Session, payload: schemas.OMAObservationCreate) -> models.OMAObservation:
    obj = models.OMAObservation(
        run_id=payload.run_id,
        mode_id=payload.mode_id,
        frequency_hz=payload.frequency_hz,
        damping_ratio=payload.damping_ratio,
        mac=payload.mac,
        quality=payload.quality,
    )
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def list_oma_observations(db: Session, structure_id: Optional[int] = None, limit: int = 200) -> List[models.OMAObservation]:
    """List OMA observations, optionally filtered by structure."""
    q = db.query(models.OMAObservation)
    if structure_id is not None:
        # Join with OMAMode to filter by structure
        q = q.join(models.OMAMode).filter(models.OMAMode.structure_id == structure_id)
    return q.order_by(models.OMAObservation.observation_id.desc()).limit(limit).all()


def upsert_pipeline_config(db: Session, service: str, key: str, value: str) -> models.PipelineConfig:
    existing = db.query(models.PipelineConfig).filter(models.PipelineConfig.service == service, models.PipelineConfig.key == key).first()
    if existing:
        existing.value = value
        db.commit()
        db.refresh(existing)
        return existing
    obj = models.PipelineConfig(service=service, key=key, value=value)
    db.add(obj)
    db.commit()
    db.refresh(obj)
    return obj


def get_pipeline_config(db: Session, service: str) -> List[models.PipelineConfig]:
    return db.query(models.PipelineConfig).filter(models.PipelineConfig.service == service).all()
