from contextlib import asynccontextmanager
from fastapi import FastAPI

from shm_shared.settings import settings

from .database import Base, engine, SessionLocal
from .routers import (
    structures,
    sensors,
    uploads,
    # damage_events,  # TODO: Re-enable when damage detection is implemented
    analysis_runs,
    cleaning_runs,
    config,
    oma,
)

from . import crud, schemas

# NOTE (prototype):
# create_all is convenient for local-first compose, but for production use Alembic migrations.
Base.metadata.create_all(bind=engine)


def _seed_default_structure() -> None:
    """Ensure the default structure exists with the expected Vicoforte metadata."""
    db = SessionLocal()
    try:
        name = settings.MONITORED_STRUCTURE
        existing = crud.get_structure_by_name(db, name)
        desired = {
            "name": name,
            "location": "Vicoforte, Italy",
            "type": "Sanctuary",
            "latitude": 44.365,
            "longitude": 7.747,
            "description": "Test structure",
        }
        if existing is None:
            crud.create_structure(db, schemas.StructureCreate(**desired))
        else:
            # Update fields if missing/outdated
            changed = False
            if existing.location != desired["location"]:
                existing.location = desired["location"]; changed = True
            if existing.structure_type != desired["type"]:
                existing.structure_type = desired["type"]; changed = True
            if existing.latitude != desired["latitude"]:
                existing.latitude = desired["latitude"]; changed = True
            if existing.longitude != desired["longitude"]:
                existing.longitude = desired["longitude"]; changed = True
            if existing.notes != desired["description"]:
                existing.notes = desired["description"]; changed = True
            if changed:
                db.commit()
    finally:
        db.close()


@asynccontextmanager
async def lifespan(app: FastAPI):
    # Startup: seed default structure
    _seed_default_structure()
    yield
    # Shutdown: nothing to do


app = FastAPI(title="SHM Metadata API", lifespan=lifespan)


@app.get("/healthz")
def health():
    return {"status": "ok"}


app.include_router(structures.router)
app.include_router(sensors.router)
app.include_router(uploads.router)
app.include_router(cleaning_runs.router)
# app.include_router(damage_events.router)  # TODO: Re-enable when damage detection is implemented
app.include_router(analysis_runs.router)
app.include_router(config.router)

app.include_router(oma.router)
