import os
import time
import requests
from watchdog.observers.polling import PollingObserver as Observer
from watchdog.events import FileSystemEventHandler

from .windows_file_utils import is_file_stable
from shm_shared.settings import settings
from shm_shared.logging import setup_logging

logger = setup_logging("ingestion.watcher")

# Use env-configured URL if present, else default.
INGEST_URL = getattr(settings, "INGEST_URL", "http://127.0.0.1:8010/ingest")
ALLOWED_SUFFIX = "_Fast.txt"  # react only to final hourly files

class Handler(FileSystemEventHandler):
    def on_created(self, event):
        if event.is_directory:
            return
        path = event.src_path

        # Only handle final data files (skip temp/partial/non-target files)
        if not path.endswith(ALLOWED_SUFFIX):
            logger.info(f"Ignoring non-target file: {path}")
            return

        logger.info(f"Detected new file: {path}")
        if is_file_stable(path, settings.STABLE_SECONDS, settings.SIZE_MIN_MB):
            logger.info(f"File is stable and >= {settings.SIZE_MIN_MB}MB: {path}")
            try:
                r = requests.post(INGEST_URL, json={"filepath": path})
                r.raise_for_status()
                logger.info(f"Ingest success: {r.json()}")
            except Exception as e:
                logger.error(f"Ingest request failed: {e}")
        else:
            logger.warning(f"File not stable or too small: {path}")

def main():
    folder = settings.HOT_FOLDER
    if not os.path.exists(folder):
        logger.warning(f"Hot folder does not exist, creating it: {folder}")
        os.makedirs(folder, exist_ok=True)

    observer = Observer()
    observer.schedule(Handler(), folder, recursive=False)
    observer.start()
    logger.info(f"Watching folder: {folder}  →  POST {INGEST_URL}")

    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        observer.stop()
    observer.join()

if __name__ == "__main__":
    main()
