from __future__ import annotations

from typing import Iterable, Sequence

import numpy as np
from influxdb_client import Point, WritePrecision

from shm_shared.influx import write_points
from shm_shared.settings import settings
from shm_shared.logging import setup_logging

logger = setup_logging("ingestion.influx_writer")


def to_points(
    structure: str,
    start_time,
    data: np.ndarray,
    sampling_hz: int,
    sensors: Sequence[str],
) -> Iterable[Point]:
    """Yield InfluxDB points for multi-channel accelerometer data (memory-safe)."""
    if data.shape[1] != len(sensors):
        raise ValueError(f"data columns ({data.shape[1]}) != sensors ({len(sensors)})")
    dt_ms = int(round(1000 / sampling_hz))
    t0_ns = int(start_time.timestamp() * 1e9)
    for i in range(data.shape[0]):
        ts_ns = t0_ns + i * dt_ms * 1_000_000
        row = data[i]
        for ch_idx, sensor in enumerate(sensors):
            yield (
                Point("accel")
                .tag("Structure", structure)
                .tag("Sensor", str(sensor))
                .field("value", float(row[ch_idx]))
                .time(ts_ns, WritePrecision.NS)
            )


def write_raw_to_influx(
    structure_name: str,
    start_time,
    data: np.ndarray,
    sampling_hz: int,
    sensors: Sequence[str],
) -> int:
    """Stream points to Influx in chunks. Returns points written."""
    max_retries = 3
    last_error = None
    for attempt in range(1, max_retries + 1):
        try:
            pts_iter = to_points(structure_name, start_time, data, sampling_hz, sensors)
            written = write_points(pts_iter, bucket=settings.RAW_BUCKET, chunk_size=50_000)
            logger.info(f"Wrote {written} points to bucket={settings.RAW_BUCKET}")
            return written
        except Exception as e:
            last_error = e
            logger.warning(f"Influx write attempt {attempt}/{max_retries} failed: {e}")
            if attempt < max_retries:
                import time
                time.sleep(attempt)  # simple backoff
    raise RuntimeError(f"Influx write failed after {max_retries} retries: {last_error}")
