from __future__ import annotations

import os
import tempfile
from datetime import timedelta
from pathlib import Path
from contextlib import asynccontextmanager

from fastapi import FastAPI, HTTPException, UploadFile, File
from pydantic import BaseModel

from shm_shared.logging import setup_logging
from shm_shared.parsing import parse_txt_file
from shm_shared.settings import settings
from shm_shared.influx import ensure_buckets
from shm_shared.metadata_client import (
    create_upload,
    mark_upload_success,
    mark_upload_failed,
    get_or_create_structure_id,
    ensure_sensors,
    get_monitored_structure_name,
)
from shm_shared.events import EventBus

from .windows_file_utils import file_size_mb
from .influx_writer import write_raw_to_influx

logger = setup_logging("ingestion.api")


@asynccontextmanager
async def lifespan(app: FastAPI):
    logger.info("Ensuring InfluxDB buckets")
    ensure_buckets()
    yield


app = FastAPI(title="SHM Ingestion Service", lifespan=lifespan)

# Per-stage stream: raw_ingested -> cleaning_worker
bus_raw = EventBus(stream=settings.STREAM_RAW_INGESTED, group=None)


class IngestRequest(BaseModel):
    filepath: str


@app.get("/healthz")
def health():
    return {"status": "ok"}


def _ingest_path(path: str) -> dict:
    if not os.path.exists(path):
        raise HTTPException(400, f"File does not exist: {path}")

    file_mb = file_size_mb(path)

    # Resolve monitored structure (UI can set this in metadata config service='system')
    structure_name = get_monitored_structure_name(settings.MONITORED_STRUCTURE)
    structure_id = get_or_create_structure_id(structure_name)

    upload_id = None
    try:
        parsed = parse_txt_file(path)
        data = parsed["data"]
        sensor_cols = parsed["sensor_cols"]
        if data.size == 0:
            raise ValueError("Parsed data is empty after trimming")

        start_time = parsed["start_time"]
        sampling_hz = int(parsed["sampling_hz"])

        # Derive end_time from sample count (more robust for pipeline windows)
        duration_sec = data.shape[0] / float(sampling_hz)
        end_time = start_time + timedelta(seconds=duration_sec)

        # Ensure sensors exist for this structure in PostgreSQL
        ensure_sensors(structure_id, list(sensor_cols))

        # Create upload record via metadata_api
        upload_id = create_upload(
            {
                "structure_id": structure_id,
                "sensor_count": len(sensor_cols),
                "filename": os.path.basename(path),
                "file_path": path,
                "file_size_mb": file_mb,
                "start_time": start_time.isoformat(),
                "end_time": end_time.isoformat(),
            }
        )

        # Write Raw series to Influx (dynamic channel names)
        points = write_raw_to_influx(structure_name, start_time, data, sampling_hz, sensors=sensor_cols)
        mark_upload_success(upload_id, points)

        # Publish pipeline event (data-plane)
        bus_raw.publish(
            {
                "type": "raw_ingested",
                "upload_id": upload_id,
                "structure_id": structure_id,
                "structure_name": structure_name,
                "start_time": start_time.isoformat(),
                "end_time": end_time.isoformat(),
                "sampling_hz": sampling_hz,
                "filename": os.path.basename(path),
                "sensor_names": list(sensor_cols),
            },
        )

        logger.info(f"Ingested upload_id={upload_id} points={points} file={path}")
        return {"upload_id": upload_id, "points": points, "sensors": list(sensor_cols)}

    except Exception as e:
        logger.exception("Ingestion failed")
        if upload_id is not None:
            try:
                mark_upload_failed(upload_id, str(e))
            except Exception:
                pass
        raise HTTPException(500, f"Ingestion failed: {e}")


@app.post("/ingest")
def ingest(req: IngestRequest):
    # Legacy path-based API (used by ingestion_watcher in docker)
    return _ingest_path(req.filepath)


@app.post("/ingest-upload")
async def ingest_upload(file: UploadFile = File(...)):
    """Upload-based ingestion (better for remote UI/manual use).

    The file is stored temporarily inside the container and then ingested.
    """
    suffix = Path(file.filename or "upload.txt").suffix or ".txt"
    with tempfile.NamedTemporaryFile(delete=False, suffix=suffix) as tmp:
        tmp_path = tmp.name
        content = await file.read()
        tmp.write(content)

    try:
        return _ingest_path(tmp_path)
    finally:
        try:
            os.remove(tmp_path)
        except Exception:
            pass
