# SHM Platform — Microservices Version 2 (Docker Compose, local-first)

This repository runs a **local-first, event-driven microservice platform** for Structural Health Monitoring (SHM) using **Docker Compose**.

**Target setup:** run everything on a single workstation (Windows + Docker Desktop), and open the **Management UI** remotely from another computer on the same network.

**Tech Stack:** FastAPI | PostgreSQL + InfluxDB | Redis | NumPy/SciPy | Plotly/Grafana | Docker

---

## Table of Contents
- [High-level architecture](#high-level-architecture-simple)
- [Services and ports](#services-and-ports-windows-host)
- [Folder layout](#folder-layout)
- [Run on Windows (Docker Desktop)](#run-on-windows-docker-desktop)
  - [Prerequisites](#prerequisites)
  - [Environment Setup](#1-environment-setup)
  - [Start the platform](#2-start-the-platform)
  - [Viewing logs](#viewing-logs)
  - [Clean shutdown](#clean-shutdown)
  - [Open the UI](#3-open-the-ui)
  - [Ingest data (automatic pipeline)](#4-ingest-data-automatic-pipeline)
- [Management UI](#management-ui-port-18020)
- [Cleaning (automatic)](#cleaning-automatic)
- [How to verify cleaning & analysis worked](#how-to-verify-cleaning--analysis-worked)
- [Operational notes](#operational-notes)
- [Troubleshooting](#troubleshooting)

---

## High-level architecture (simple)

**Data plane (time-series):**
1) A `.txt` file appears in the hot folder
2) Ingestion parses it and writes **Raw** signals to InfluxDB
3) Cleaning worker consumes an event and writes **Cleaned** signals to InfluxDB
4) Analysis worker consumes an event and writes:
   - **OMA_Data** bucket: modal properties (frequency, damping, MAC values)
   - **Analyzed_Data** bucket: reserved for other analysis types (extensible)

**Control / metadata plane:**
- Postgres stores: structures, sensors (auto-detected), uploads, and cleaning/analysis run logs
- Management UI lets you change cleaning + analysis parameters and run manual actions

**Event bus:** Redis Streams with **per-stage streams** (prevents pipeline loss):
- `shm_raw_ingested` → cleaning
- `shm_cleaned_ready` → analysis
- `shm_analysis_done` → downstream consumers (future)

---

## Services and ports (Windows host)

| Service | Host port | Inside container | Remote Access |
|---|---:|---:|---|
| **Management UI** | **18020** | 8020 | ✅ **Yes** (binds to all interfaces) |
| **Grafana** | **3001** | 3000 | ✅ **Yes** (binds to `0.0.0.0`) |
| InfluxDB | 8086 | 8086 | ❌ Localhost only (`127.0.0.1`) |
| Postgres | 15432 | 5432 | ❌ Localhost only (`127.0.0.1`) |

> APIs (ingestion/cleaning/metadata) are internal to the Docker network by default.

### Accessing from other computers on the same network (LAN)
From any device on your local network:
- **Management UI**: `http://<WORKSTATION_IP>:18020`
- **Grafana**: `http://<WORKSTATION_IP>:3001`

Replace `<WORKSTATION_IP>` with your workstation's LAN IP (find it with `ipconfig` on Windows).

**Firewall setup:** Allow inbound TCP ports **18020** and **3001** in Windows Defender Firewall.

### Accessing remotely over the internet
⚠️ **Not recommended for production.** For internet access, you would need:
- Port forwarding on your router (ports 18020 and 3001 → workstation IP)
- Static IP or Dynamic DNS
- **Security considerations:** Add authentication, HTTPS/TLS, and firewall rules

For production internet access, consider deploying to a cloud platform with proper security.

---

## Folder layout

These folders are mounted into containers:

- `StructuralX_Data/` → **incoming hot folder** (watcher monitors this)
- `Temp_data/` → temporary/internal artifacts and **sample raw files**
- `Cleaned_Text/` → outputs from **manual file-cleaning** actions

### Sample data change in Version 2

To test ingestion, copy/move one sample file from `Temp_data/` into `StructuralX_Data/`.

---

# Run on Windows (Docker Desktop)

## Prerequisites
- Windows 10/11
- Docker Desktop installed and running (WSL2 backend recommended)

## 1) Environment Setup
The `.env` file is already included with defaults for local-first use.

To customize, copy from the example:
```powershell
copy .env.example .env
```

Key settings: `HOT_FOLDER=/data/incoming`, bucket names, and service URLs.

## 2) Start the platform

```powershell
docker compose up --build
```

Wait until you see:
- the Influx init container reporting buckets were ensured
- healthchecks passing for the main services

When to use each command:
- `docker compose up --build`	You modified the Dockerfile or your source code.
- `docker compose up --build management_ui` To rebuild and restart only the management UI container for applying a fix or change.
- `docker compose start` Similar to the Play button in Docker Desktop (resumes containers that already exist but are in a "Stopped" or "Exited" state.)
- `docker compose up`	You want to start the app using existing images. `up` is smarter than `start`: It checks if the containers exist; if they don't, it creates them. If they do, it starts them.
- `docker compose up -d`	You want the app running but need your terminal back.
Use `docker compose logs -f` To view logs while running in detached mode (-d)

**Important:** If you start containers using the Docker Desktop Play button (same as `docker compose start`), code changes are **not** picked up. Rebuild to apply updates:

### Viewing logs
- `docker compose logs --tail=100` shows the most recent 100 lines from all services.
- `docker compose logs -f ingestion_api metadata_api` follows logs for selected services.
- Add `--tail=100` to start with the last 100 lines (for example: `docker compose logs -f --tail=100 ingestion_api metadata_api`).

## Clean shutdown
- `docker compose stop`	You want to pause the app without deleting the containers.
- `docker compose down`	You want to stop the app and remove the containers/network.
-  `docker compose down -v` You want to stop the app and remove the containers and volumes (this deletes Postgres/Influx persistent data).
- `docker compose down --rmi all` You want to stop the app and remove containers, networks, **and images** (forces a clean rebuild next time).

## 3) Open the UI
- On the workstation: `http://localhost:18020`
- From another PC on the same network: `http://<WORKSTATION_IP>:18020`

If remote access fails:
- allow inbound TCP **18020** in Windows Defender Firewall
- confirm you are using the workstation LAN IP (run `ipconfig`)

## 4) Ingest data (automatic pipeline)
Drop a `.txt` file into:
- `StructuralX_Data/`

The watcher detects the completed file and calls ingestion automatically.

**Quick test using the provided sample files:**

```powershell
# example: move a sample into the hot folder
move .\Temp_data\2025_10_26_05_04_18_Fast.txt .\StructuralX_Data\
```

---

# Management UI (port 18020)

**Dashboard** (`/`):
- View live cleaning/analysis parameters and structure info
- Edit cleaning settings (HP, LP, downsample, filter order, detrend)
- Edit analysis settings (frequency range, max modes, MAC threshold, tolerances)
- View sensor list per structure
- Inspect recent file cleaning runs

**Manual file cleaning** (`/manual`):
- Upload a `.txt` file for preview
- Visualize raw signal, FFT spectrum, and cleaned signal
- Select specific channels to clean
- Download cleaned output (`.txt`)

---

# Cleaning (automatic)

Cleaning runs automatically after ingestion.

## What cleaning does (per channel)
1) mean removal
2) polynomial detrending (default: linear)
3) Butterworth band-pass filter (default: 0.5–15 Hz, zero-phase)
4) downsampling (default factor 2)

## InfluxDB & Cleaning configuration
Optional settings (defaults work out of the box):

```dotenv
# InfluxDB buckets (separate for scalability)
RAW_BUCKET=Raw_Data          # Raw signals
CLEANED_BUCKET=Cleaned_Data  # Filtered signals
OMA_BUCKET=OMA_Data          # Modal properties
ANALYZED_BUCKET=Analyzed_Data # Future analysis types

# Cleaning parameters
CLEANING_HP=0.5
CLEANING_LP=15.0
CLEANING_DOWNSAMPLE=2
CLEANING_FILTER_ORDER=4
CLEANING_DETREND_ORDER=1
```

## How to verify cleaning & analysis worked
### InfluxDB (on the workstation)
InfluxDB is bound to localhost by default:
- open `http://localhost:8086`
- query `Raw_Data`, `Cleaned_Data`, and `OMA_Data` buckets for recent data

### Postgres (from a shell open at the project directory)
You can check run logs via docker exec:

```powershell
# open psql shell in the postgres container by POSTGRES_USER and POSTGRES_DB
docker compose exec postgres psql -U postgres -d shm
or
docker compose exec postgres psql -U shm -d SHM_ProjectDB

# then list tables:
\dt
# list schemas:
\dn
# describe a table:
\d structures
# to check if the seed structure exists
SELECT structure_id, name, location, structure_type, latitude, longitude, notes
FROM structures
ORDER BY structure_id;

# then run:
SELECT * FROM cleaning_runs ORDER BY cleaning_id DESC LIMIT 5;
SELECT * FROM analysis_runs  ORDER BY run_id DESC LIMIT 5;
SELECT * FROM structures;
SELECT * FROM sensors;
SELECT structure_id, name, location, structure_type, latitude, longitude, notes FROM structures ORDER BY structure_id;

# to close the view:
q
```

---

# Operational notes

## Do I need to remove the previous Docker image for port compatibility?
**No.** Port conflicts are caused by **running containers** (or other apps) already using the same host ports — not by old images.

If you see “port is already allocated”, do one of these:
1) Stop the old stack:
```powershell
# run this from the old project folder
docker compose down
```
2) Or stop conflicting containers in Docker Desktop
3) Or change the published ports in this repo’s `docker-compose.yml` (e.g., change `18020:8020` to another free port)
---

# Troubleshooting

## UI loads locally but not remotely
- Ensure `management_ui` is published on `0.0.0.0` (it is in this compose)
- Allow inbound TCP 18020 in Windows Firewall
- Use the workstation LAN IP (not `localhost`) from the remote PC

## Influx buckets missing
Buckets are created on ingestion service startup: `Cleaned_Data`, `Analyzed_Data`, and `OMA_Data` (Raw_Data is created by docker-compose).
If buckets are missing, verify `.env` org/token values match docker-compose.yml settings and check ingestion service logs.

## Watcher doesn’t detect files
- Make sure you are copying into `StructuralX_Data/`
- Wait until the file copy is complete (large files can take time)
- Check watcher logs:
```powershell
docker compose logs -f ingestion_watcher
```

## Port conflicts
If Docker reports a port is in use:
- stop the other stack (`docker compose down`) or
- change host ports in `docker-compose.yml`.
