const express = require('express');
const helmet = require('helmet');
const morgan = require('morgan');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

// Import routes
const teamRoutes = require('./routes/aboutus');
const useCaseRoutes = require('./routes/usecase');
const researchRoutes = require('./routes/research');
const infrastructureRoutes = require('./routes/infrastructure');

const app = express();
const PORT = process.env.PORT || 3000;

// Logging Setup
const logDirectory = path.join(process.cwd(), 'logs');

const accessLogStream = fs.createWriteStream(
  path.join(logDirectory, 'inNuCE_Web.log'),
  { flags: 'a' }
);

// 
if (process.env.NODE_ENV === 'development') {
  app.use(morgan('dev')); // colorful log
} else {
  app.use(morgan('combined', { stream: accessLogStream })); // write into the logs txt
}

// Middleware
app.use(helmet({
  crossOriginResourcePolicy: { policy: "cross-origin" }
}));
app.use(express.json());

// Static Resource
// console.log('Serving static files from:', path.join(__dirname, 'static'));
app.use('/static', express.static(path.join(__dirname, 'static')));

// API Routes
app.use('/aboutus', teamRoutes);
app.use('/usecase', useCaseRoutes);
app.use('/research', researchRoutes);
app.use('/infrastructure', infrastructureRoutes);

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ status: 'OK', timestamp: new Date().toISOString() });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({ error: 'Something went wrong!' });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ error: 'Endpoint not found' });
});

app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

module.exports = app;
