const express = require('express');
const router = express.Router();
const { query } = require('../database');

// GET /api/use-cases - Fetch all use cases
router.get('/cards', async (req, res) => {
  try {
    const result = await query(`
      SELECT
            uc.id::text AS id,
            uc.title,
            uc.description,
            uc.image_src,

            COALESCE(
                (SELECT json_agg(k.name)
                 FROM usecase_schema.keyword_usecase ku
                 JOIN usecase_schema.keyword k ON ku.keyword_id = k.id
                 WHERE ku.usecase_id = uc.id),
                '[]'::json
            ) AS keywords,

            COALESCE(
                (SELECT json_agg(f.name)
                 FROM usecase_schema.framework_usecase fu
                 JOIN usecase_schema.framework f ON fu.framework_id = f.id
                 WHERE fu.usecase_id = uc.id),
                '[]'::json
            ) AS frameworks,

            COALESCE(
                (SELECT json_agg(m.name)
                 FROM usecase_schema.model_usecase mu
                 JOIN usecase_schema.model m ON mu.model_id = m.id
                 WHERE mu.usecase_id = uc.id),
                '[]'::json
            ) AS models,

            COALESCE(
                (SELECT json_agg(h.name)
                 FROM usecase_schema.hardware_usecase hu
                 JOIN usecase_schema.hardware h ON hu.hardware_id = h.id
                 WHERE hu.usecase_id = uc.id),
                '[]'::json
            ) AS hardware

      FROM
            usecase_schema.usecase uc
    `);

    // transform data
    const baseUrl = process.env.BASE_URL;
    const transformedData = result.rows.map(usecase => {
      return {
        id: usecase.id,
        title: usecase.title,
        description: usecase.description,
        keywords: usecase.keywords,
        frameworks: usecase.frameworks,
        models: usecase.models,
        hardware: usecase.hardware,
        image_src: usecase.image_src ? `${baseUrl}/assets/${usecase.image_src}` : null
      }
    })

    res.json({
      success: true,
      data: transformedData,
      count: transformedData.length
    });
  } catch (error) {
    console.error('Error fetching use cases:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch use cases cards'
    });
  }
});

// GET /api/filter - Fetch filter options
router.get('/filter', async (req, res) => {
  try {
    const [frameworks, models, keywords, hardware] = await Promise.all([
      query('SELECT name FROM usecase_schema.framework ORDER BY name ASC'),
      query('SELECT name FROM usecase_schema.model ORDER BY name ASC'),
      query('SELECT name FROM usecase_schema.keyword ORDER BY name ASC'),
      query('SELECT name FROM usecase_schema.hardware ORDER BY name ASC')
    ]);

    const formatData = (title, result) => ({
      title,
      tags: result.rows.map(row => row.name)
    });

    const transformedData = [
      formatData('FRAMEWORK', frameworks),
      formatData('MODEL', models),
      formatData('KEYWORDS', keywords),
      formatData('HARDWARE', hardware)
    ];

    res.json({
      success: true,
      data: transformedData,
      count: transformedData.length
    });
  } catch (error) {
    console.error('Error fetching filter options:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch filter options'
    });
  }
});

// GET /api/use-cases - Fetch specific use cases
router.get('/:slug', async (req, res) => {
  const { slug } = req.params;
  try {
    const result = await query(`
      SELECT
            uc.id::text AS id,
            ud.title,
            ud.caption,
            ud.main_image,
            ud.publications,
            ud.source_code,
            ud.dataset,

            COALESCE(
                (SELECT json_agg(json_build_object(
                    'title', p.title,
                    'content', p.content,
                    'image', p.image
                ) ORDER BY p.order_index)
                 FROM usecase_schema.paragraph p
                 WHERE p.usecase_detail_id = ud.id),
                '[]'::json
            ) AS paragraphs

      FROM
            usecase_schema.usecase uc
      LEFT JOIN
            usecase_schema.usecase_detail ud ON uc.usecase_detail_id = ud.id
      WHERE
            uc.id = $1::uuid; 
    `, [slug]);

    if (result.rows.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Use case not found'
      });
    }

    // transform data
    const baseUrl = process.env.BASE_URL;
    const usecase = result.rows[0];

    const transformedData = {
      id: usecase.id,
      title: usecase.title,
      caption: usecase.caption,
      main_image: usecase.main_image ? `${baseUrl}/assets/${usecase.main_image}` : "",
      paragraphs: usecase.paragraphs.map(p => ({
        ...p,
        image: p.image ? `${baseUrl}/assets/${p.image}` : ""
      })),
      publications: usecase.publications,
      source_code: usecase.source_code,
      dataset: usecase.dataset
    };

    res.json({
      success: true,
      data: transformedData,
      count: transformedData.length
    });

  } catch (error) {
    console.error('Error fetching use case details:', error);
    res.status(500).json({
      success: false,
      error: `Failed to fetch use case details: ${error.message}`
    });
  }
});



module.exports = router;
