const express = require('express');
const router = express.Router();
const { query } = require('../database');

// GET /api/research - Fetch all research publications
router.get('/', async (req, res) => {
    try {
        const result = await query(`
      SELECT 
        title, 
        publication_year, 
        url, 
        keyword,
        audio_url,
        video_url,
        mental_map_url,
        report_url,
        flashcard_url,
        quiz_url,
        infografica_url,
        presentation_url,
        datatable_url
      FROM publication_schema.publications pp
      WHERE pp.reviewed = true
      ORDER BY publication_year DESC
    `);
        const baseUrl = process.env.BASE_URL;
        const transformedData = result.rows.map(row => {
            // Parse keywords
            let keywordsList = [];
            if (row.keyword) {
                keywordsList = row.keyword.split(',').filter(k => k.trim().length > 0);
            }
            return {
                title: row.title,
                date: parseInt(row.publication_year, 10) || 0,
                link: row.url || null,
                keywords: keywordsList,
                audio_url: row.audio_url ? `${baseUrl}/assets/${row.audio_url}` : null,
                video_url: row.video_url ? `${baseUrl}/assets/${row.video_url}` : null,
                mental_map_url: row.mental_map_url ? `${baseUrl}/assets/${row.mental_map_url}` : null,
                report_url: row.report_url ? `${baseUrl}/assets/${row.report_url}` : null,
                flashcard_url: row.flashcard_url ? `${baseUrl}/assets/${row.flashcard_url}` : null,
                quiz_url: row.quiz_url ? (row.quiz_url.startsWith('http') ? row.quiz_url : `${baseUrl}/assets/${row.quiz_url}`) : null,
                infografica_url: row.infografica_url ? `${baseUrl}/assets/${row.infografica_url}` : null,
                presentation_url: row.presentation_url ? `${baseUrl}/assets/${row.presentation_url}` : null,
                datatable_url: row.datatable_url ? `${baseUrl}/assets/${row.datatable_url}` : null
            };
        });
        res.json({
            success: true,
            data: transformedData,
            count: transformedData.length
        });
    } catch (error) {
        console.error('Error fetching research publications:', error);
        res.status(500).json({
            success: false,
            error: 'Failed to fetch research publications'
        });
    }
});

module.exports = router;
