"""
File handler utility for saving generated resources.
"""
import os
import json
from datetime import datetime
from typing import Any, Dict


class FileHandler:
    """Handle file operations for generated resources."""
    
    def __init__(self, output_dir: str):
        """
        Initialize file handler.
        
        Args:
            output_dir: Base output directory
        """
        self.output_dir = output_dir
        self._ensure_directories()
    
    def _ensure_directories(self):
        """Ensure output directory exists."""
        os.makedirs(self.output_dir, exist_ok=True)
    
    def save_text(self, content: str, resource_type: str, filename: str = None) -> str:
        """
        Save text content to file.
        
        Args:
            content: Text content to save
            resource_type: Type of resource (used for default naming if filename logic requires it, but not for directory)
            filename: Optional custom filename
            
        Returns:
            Path to saved file
        """
        if not filename:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            filename = f"{resource_type}_{timestamp}.txt"
        
        filepath = os.path.join(self.output_dir, filename)
        
        with open(filepath, 'w', encoding='utf-8') as f:
            f.write(content)
        
        return filepath
    
    def save_json(self, data: Dict[str, Any], resource_type: str, filename: str = None) -> str:
        """
        Save JSON data to file.
        
        Args:
            data: Dictionary to save as JSON
            resource_type: Type of resource
            filename: Optional custom filename
            
        Returns:
            Path to saved file
        """
        if not filename:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            filename = f"{resource_type}_{timestamp}.json"
        
        filepath = os.path.join(self.output_dir, filename)
        
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=2, ensure_ascii=False)
        
        return filepath
    
    def save_binary(self, data: bytes, resource_type: str, filename: str, extension: str = None) -> str:
        """
        Save binary data to file.
        
        Args:
            data: Binary data to save
            resource_type: Type of resource
            filename: Filename (without extension)
            extension: File extension (e.g., 'mp3', 'mp4')
            
        Returns:
            Path to saved file
        """
        if extension and not filename.endswith(f'.{extension}'):
            filename = f"{filename}.{extension}"
        
        filepath = os.path.join(self.output_dir, filename)
        
        with open(filepath, 'wb') as f:
            f.write(data)
        
        return filepath
    
    def get_resource_path(self, resource_type: str, filename: str) -> str:
        """
        Get full path for a resource file.
        
        Args:
            resource_type: Type of resource (ignored for path, kept for interface compatibility)
            filename: Filename
            
        Returns:
            Full file path
        """
        return os.path.join(self.output_dir, filename)
    
    def list_resources(self, resource_type: str) -> list:
        """
        List all files in the output directory that start with resource_type.
        
        Args:
            resource_type: Type of resource
            
        Returns:
            List of filenames
        """
        dir_path = self.output_dir
        if os.path.exists(dir_path):
            return [f for f in os.listdir(dir_path) 
                    if os.path.isfile(os.path.join(dir_path, f)) and f.startswith(resource_type + "_")]
        return []
