"""
Text-to-Speech plugin for converting scripts to audio.
"""
import os
import re
import asyncio
import edge_tts
import imageio_ffmpeg
from gtts import gTTS # Keep as fallback
from pydub import AudioSegment

# Configure pydub to use the ffmpeg binary from imageio-ffmpeg
AudioSegment.converter = imageio_ffmpeg.get_ffmpeg_exe()
AudioSegment.ffmpeg = AudioSegment.converter
AudioSegment.ffprobe = AudioSegment.converter

class TextToSpeech:
    """Convert text to speech audio files using Edge TTS (high quality) or gTTS (fallback)."""
    
    def __init__(self, language='en'):
        self.language = language
        # Default voice for single speaker
        self.default_voice = "en-US-GuyNeural"
        # Podcast voices
        self.host_voice = "en-US-GuyNeural"   # Male
        self.guest_voice = "en-US-AriaNeural" # Female
    
    async def _generate_edge_audio(self, text: str, output_path: str, voice: str):
        """Generate audio using Edge TTS."""
        communicate = edge_tts.Communicate(text, voice)
        await communicate.save(output_path)

    def generate_audio(self, text: str, output_path: str, format='mp3'):
        """
        Sync wrapper for simple TTS (used by VideoGenerator).
        Uses Edge TTS default voice.
        """
        try:
            asyncio.run(self._generate_edge_audio(text, output_path, self.default_voice))
        except Exception as e:
            print(f"Edge TTS failed ({e}), falling back to gTTS...")
            tts = gTTS(text=text, lang=self.language, slow=False)
            tts.save(output_path)

    async def generate_podcast_audio(self, script_data: list, output_path: str):
        """
        Generate dual-speaker podcast audio from JSON script using MoviePy for concatenation.
        """
        # Import inside method to avoid circular imports or heavy load if not used
        from moviepy import AudioFileClip, concatenate_audioclips
        
        temp_files = []
        clips = []
        
        # Create a temp dir for segments (unique per file to avoid collisions)
        temp_dir = output_path + "_segments"
        os.makedirs(temp_dir, exist_ok=True)

        try:
            print("Generating audio segments...")
            for i, turn in enumerate(script_data):
                speaker = turn.get('speaker', 'Host')
                text = turn.get('text', '')
                if not text:
                    continue
                
                # Select voice
                voice = self.host_voice if "Host" in speaker else self.guest_voice
                
                # Generate segment
                seg_filename = f"seg_{i:03d}.mp3"
                seg_path = os.path.join(temp_dir, seg_filename)
                
                try:
                    await self._generate_edge_audio(text, seg_path, voice)
                    temp_files.append(seg_path)
                    
                    # Create Clip
                    clip = AudioFileClip(seg_path)
                    clips.append(clip)
                    
                except Exception as e:
                    print(f"Error generating segment {i}: {e}")
            
            # Export final
            if clips:
                print(f"Concatenating {len(clips)} segments to {output_path}...")
                final_audio = concatenate_audioclips(clips)
                final_audio.write_audiofile(output_path, fps=44100, logger=None)
                print("Audio export validation: Success.")
            else:
                raise ValueError("No audio segments generated successfully.")
            
        finally:
            # Close clips to release file handles
            for clip in clips:
                try:
                    clip.close()
                except:
                    pass

            # Cleanup temp files
            import shutil
            if os.path.exists(temp_dir):
                shutil.rmtree(temp_dir)

    def text_to_audio_with_pauses(self, text: str, output_path: str, pause_duration=300):
        """Legacy method if needed."""
        self.generate_audio(text, output_path)
