"""
Slide Generator Plugin.
Generates static slide images using PIL (Pillow) with a modern aesthetic.
"""
import os
import textwrap
import random
from PIL import Image, ImageDraw, ImageFont
from typing import List, Tuple

class SlideGenerator:
    """Generates slide images for video presentation."""
    
    def __init__(self, width=1920, height=1080):
        self.width = width
        self.height = height
        # Modern color palette (Deep Blue/Purple gradients)
        self.colors = [
            (15, 23, 42),  # Slate 900
            (30, 41, 59),  # Slate 800
            (88, 28, 135), # Purple 900
            (49, 46, 129), # Indigo 900
            (67, 56, 202), # Indigo 700
        ]
        self.font_path = self._get_font_path()
        self.title_font_size = 80
        self.content_font_size = 50
        
    def _get_font_path(self):
        """Try to find a good sans-serif font."""
        candidates = [
            # macOS
            '/System/Library/Fonts/Supplemental/Arial Rounded Bold.ttf',
            '/System/Library/Fonts/Helvetica.ttc',
            '/System/Library/Fonts/Avenir.ttc',
             # Windows
            'C:\\Windows\\Fonts\\arial.ttf',
            'C:\\Windows\\Fonts\\seguiemj.ttf',
             # Linux
            '/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf'
        ]
        for c in candidates:
            if os.path.exists(c):
                return c
        return None

    def _create_gradient_background(self):
        """Create a gradient background."""
        base = Image.new('RGB', (self.width, self.height), self.colors[0])
        top = Image.new('RGB', (self.width, self.height), self.colors[0])
        mask = Image.new('L', (self.width, self.height))
        mask_data = []
        for y in range(self.height):
            mask_data.extend([int(255 * (y / self.height))] * self.width)
        mask.putdata(mask_data)
        
        # Blend random color
        c2 = random.choice(self.colors[1:])
        top.paste(c2, [0, 0, self.width, self.height])
        base.paste(top, (0, 0), mask)
        return base

    def generate_slide(self, title: str, content: List[str], output_path: str):
        """
        Generate a single slide image.
        
        Args:
            title: Slide title
            content: List of bullet points
            output_path: Where to save the image
        """
        img = self._create_gradient_background()
        draw = ImageDraw.Draw(img)
        
        # Load fonts
        try:
            if self.font_path:
                title_font = ImageFont.truetype(self.font_path, self.title_font_size)
                content_font = ImageFont.truetype(self.font_path, self.content_font_size)
            else:
                title_font = ImageFont.load_default()
                content_font = ImageFont.load_default()
        except:
             title_font = ImageFont.load_default()
             content_font = ImageFont.load_default()

        # Draw Title (Centered top)
        # Handle title wrapping
        margin = 100
        title_width = self.width - 2 * margin
        title_lines = textwrap.wrap(title, width=30) # Rough char approximation
        
        y_text = 100
        for line in title_lines:
            # Calculate text width safely
            left, top, right, bottom = draw.textbbox((0, 0), line, font=title_font)
            w = right - left
            h = bottom - top
            draw.text(((self.width - w) / 2, y_text), line, font=title_font, fill='white')
            y_text += h + 20
            
        # Draw underline
        draw.line((margin, y_text + 20, self.width - margin, y_text + 20), fill=(56, 189, 248), width=5) # Cyan underline
        
        # Draw Content
        y_text += 100
        for item in content:
            # Bullet point symbol
            draw.text((margin, y_text), "•", font=content_font, fill=(56, 189, 248))
            
            # Wrap content content
            content_lines = textwrap.wrap(item, width=50)
            for line in content_lines:
                draw.text((margin + 60, y_text), line, font=content_font, fill='white')
                y_text += self.content_font_size + 15
            
            y_text += 30 # Extra space between bullets
            
        img.save(output_path)
        return output_path
