"""
Presentation builder plugin for creating PowerPoint files from JSON specifications.
"""
from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.enum.text import PP_ALIGN


class PresentationBuilder:
    """Build PowerPoint presentations from JSON specifications."""
    
    def __init__(self):
        """Initialize presentation builder."""
        self.prs = None
    
    def create_from_json(self, presentation_data: dict, output_path: str):
        """
        Create PowerPoint file from presentation JSON.
        
        Args:
            presentation_data: Presentation specification dict
            output_path: Path to save .pptx file
        """
        self.prs = Presentation()
        
        # Set slide size to standard 16:9
        self.prs.slide_width = Inches(10)
        self.prs.slide_height = Inches(7.5)
        
        slides = presentation_data.get('slides', [])
        
        for slide_data in slides:
            self._add_slide(slide_data)
        
        # Save presentation
        self.prs.save(output_path)
        return output_path
    
    def _add_slide(self, slide_data: dict):
        """Add a slide to the presentation."""
        slide_type = slide_data.get('type', 'content')
        
        if slide_type == 'title':
            self._add_title_slide(slide_data)
        elif slide_type == 'agenda':
            self._add_bullet_slide(slide_data)
        elif slide_type == 'content':
            self._add_content_slide(slide_data)
        elif slide_type == 'data':
            self._add_data_slide(slide_data)
        else:
            # Default to content slide
            self._add_content_slide(slide_data)
    
    def _add_title_slide(self, slide_data: dict):
        """Add a title slide."""
        slide_layout = self.prs.slide_layouts[0]  # Title slide layout
        slide = self.prs.slides.add_slide(slide_layout)
        
        title = slide.shapes.title
        subtitle = slide.placeholders[1]
        
        title.text = slide_data.get('title', '')
        subtitle.text = slide_data.get('subtitle', '')
    
    def _add_bullet_slide(self, slide_data: dict):
        """Add a slide with bullet points."""
        slide_layout = self.prs.slide_layouts[1]  # Title and content layout
        slide = self.prs.slides.add_slide(slide_layout)
        
        title = slide.shapes.title
        title.text = slide_data.get('title', '')
        
        content_placeholder = slide.placeholders[1]
        text_frame = content_placeholder.text_frame
        
        for item in slide_data.get('content', []):
            p = text_frame.add_paragraph()
            p.text = str(item)
            p.level = 0
    
    def _add_content_slide(self, slide_data: dict):
        """Add a general content slide."""
        slide_layout = self.prs.slide_layouts[5]  # Blank layout
        slide = self.prs.slides.add_slide(slide_layout)
        
        # Add title
        left = Inches(0.5)
        top = Inches(0.5)
        width = Inches(9)
        height = Inches(1)
        
        title_box = slide.shapes.add_textbox(left, top, width, height)
        title_frame = title_box.text_frame
        title_frame.text = slide_data.get('title', '')
        
        # Format title
        for paragraph in title_frame.paragraphs:
            paragraph.font.size = Pt(32)
            paragraph.font.bold = True
        
        # Add content
        if slide_data.get('content'):
            left = Inches(0.5)
            top = Inches(1.8)
            width = Inches(9)
            height = Inches(5)
            
            content_box = slide.shapes.add_textbox(left, top, width, height)
            content_frame = content_box.text_frame
            
            for item in slide_data.get('content', []):
                p = content_frame.add_paragraph()
                p.text = f"• {item}"
                p.font.size = Pt(18)
                p.space_after = Pt(12)
    
    def _add_data_slide(self, slide_data: dict):
        """Add a slide focused on data/charts."""
        # For now, similar to content slide
        # In production, you'd add actual charts using python-pptx chart capabilities
        self._add_content_slide(slide_data)
        
        # Add note about visualization
        layout = self.prs.slide_layouts[5]
        slide = self.prs.slides[-1]  # Get the slide we just added
        
        # Add a note text box
        left = Inches(0.5)
        top = Inches(6.5)
        width = Inches(9)
        height = Inches(0.8)
        
        note_box = slide.shapes.add_textbox(left, top, width, height)
        note_frame = note_box.text_frame
        
        viz_note = slide_data.get('data_visualization', {})
        if viz_note:
            note_text = f"Visualization: {viz_note.get('type', 'Chart')} - {viz_note.get('description', '')}"
            note_frame.text = note_text
            note_frame.paragraphs[0].font.size = Pt(10)
            note_frame.paragraphs[0].font.italic = True
