"""
Diagram generator plugin for creating visual diagrams from DOT format.
"""
import os
from graphviz import Source
from PIL import Image, ImageDraw, ImageFont
from utils.logger import get_logger

logger = get_logger(__name__)


class DiagramGenerator:
    """Generate visual diagrams from text descriptions."""
    
    def __init__(self):
        """Initialize diagram generator."""
        pass
    
    def generate_from_dot(self, dot_code: str, output_path: str, format='png'):
        """
        Generate diagram from GraphViz DOT code.
        
        Args:
            dot_code: DOT format graph description
            output_path: Path to save the generated image
            format: Output format (default: 'png')
        """
        try:
            # Create graph from DOT code
            graph = Source(dot_code)
            
            # Render to file
            # Remove extension from output_path if present
            base_path = output_path.rsplit('.', 1)[0]
            
            graph.render(base_path, format=format, cleanup=True)
            
            # graphviz.render creates filename.format, so rename if needed
            rendered_file = f"{base_path}.{format}"
            if rendered_file != output_path:
                os.rename(rendered_file, output_path)
            
            return output_path
            
        except Exception as e:
            logger.error(f"Error generating diagram: {e}")
            raise
    
    def create_simple_mindmap_image(self, mindmap_text: str, output_path: str):
        """
        Create a simple visual representation of a text-based mind map.
        This is a fallback when DOT code is not available.
        
        Args:
            mindmap_text: Text representation of mind map
            output_path: Path to save image
        """
        # Create a simple image with the mind map text
        # This is a basic implementation - for production, you'd want more sophisticated rendering
        
        width, height = 1200, 1600
        img = Image.new('RGB', (width, height), color='white')
        draw = ImageDraw.Draw(img)
        
        try:
            font = ImageFont.truetype("/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf", 14)
            title_font = ImageFont.truetype("/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf", 20)
        except:
            font = ImageFont.load_default()
            title_font = font
        
        # Draw title
        draw.text((20, 20), "Mind Map", fill='black', font=title_font)
        
        # Draw mind map text (wrapped)
        y_offset = 60
        max_width = width - 40
        
        lines = mindmap_text.split('\n')
        for line in lines[:100]:  # Limit to 100 lines
            if y_offset > height - 40:
                break
            
            # Simple text wrapping
            words = line.split()
            current_line = ''
            
            for word in words:
                test_line = current_line + ' ' + word if current_line else word
                # Rough width calculation
                if len(test_line) * 8 < max_width:  # Approx 8 pixels per char
                    current_line = test_line
                else:
                    if current_line:
                        draw.text((20, y_offset), current_line, fill='black', font=font)
                        y_offset += 20
                    current_line = word
            
            if current_line:
                draw.text((20, y_offset), current_line, fill='black', font=font)
                y_offset += 20
        
        img.save(output_path)
        return output_path
    
    def generate_hierarchical_diagram(self, structure: dict, output_path: str):
        """
        Generate a hierarchical diagram from a nested dictionary structure.
        
        Args:
            structure: Dict representing hierarchy
            output_path: Path to save diagram
        """
        # Convert structure to DOT format
        dot_code = self._dict_to_dot(structure)
        return self.generate_from_dot(dot_code, output_path)
    
    def _dict_to_dot(self, structure: dict, graph_name='Diagram') -> str:
        """Convert nested dictionary to DOT format."""
        lines = [
            f'digraph {graph_name} {{',
            '  rankdir=TB;',
            '  node [shape=box, style=rounded];',
            ''
        ]
        
        def add_nodes(data, parent_id=None, counter=[0]):
            if isinstance(data, dict):
                for key, value in data.items():
                    counter[0] += 1
                    node_id = f'node{counter[0]}'
                    lines.append(f'  {node_id} [label="{key}"];')
                    
                    if parent_id:
                        lines.append(f'  {parent_id} -> {node_id};')
                    
                    if isinstance(value, (dict, list)):
                        add_nodes(value, node_id, counter)
            
            elif isinstance(data, list):
                for item in data:
                    if isinstance(item, (dict, list)):
                        add_nodes(item, parent_id, counter)
                    else:
                        counter[0] += 1
                        node_id = f'node{counter[0]}'
                        lines.append(f'  {node_id} [label="{item}"];')
                        if parent_id:
                            lines.append(f'  {parent_id} -> {node_id};')
        
        add_nodes(structure)
        lines.append('}')
        
        return '\n'.join(lines)
