"""
Video generator - creates video scripts with scene descriptions.
"""
import json
import os
import shutil
import uuid
from typing import Dict, Any, List
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.video_prompt import VIDEO_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger
from plugins.slide_generator import SlideGenerator
from plugins.text_to_speech import TextToSpeech
from moviepy import ImageClip, AudioFileClip, concatenate_videoclips

logger = get_logger(__name__)


class VideoGenerator(BaseGenerator):
    """Generate presentation video from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.script_data = None # List of dicts (slides)
        self.slide_gen = SlideGenerator()
        self.tts = TextToSpeech()
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate video script (JSON) using LLM.
        
        Returns:
            Dictionary with script and metadata
        """
        logger.info("Generating video script structure...")
        
        # Generate script using LLM
        response = self.generate_with_prompt(VIDEO_PROMPT)
        
        # Clean response to ensure valid JSON
        cleaned_response = response.strip()
        if cleaned_response.startswith('```json'):
            cleaned_response = cleaned_response[7:]
        if cleaned_response.endswith('```'):
            cleaned_response = cleaned_response[:-3]
            
        try:
            self.script_data = json.loads(cleaned_response)
        except json.JSONDecodeError as e:
            logger.error(f"Failed to parse LLM JSON output: {e}")
            logger.error(f"Raw output start: {cleaned_response[:500]}...") # Log first 500 chars
            # Fallback strategy: try to find JSON array if wrapped in other text
            try:
                start = cleaned_response.find('[')
                end = cleaned_response.rfind(']') + 1
                if start != -1 and end != 0:
                     potential_json = cleaned_response[start:end]
                     self.script_data = json.loads(potential_json)
                     logger.info("Recovered JSON from text wrapper.")
                else:
                    raise ValueError("Could not find JSON array in output")
            except Exception as e2:
                logger.error(f"Fallback JSON parsing failed: {e2}")
                logger.error(f"Full problematic response: {response}")
                raise ValueError(f"LLM did not return valid JSON. Error: {e}. Raw (snippet): {cleaned_response[:100]}")
        
        return {
            'type': 'video',
            'script_data': self.script_data,
            'generated_at': datetime.now().isoformat(),
            'scene_count': len(self.script_data)
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Generate assets and render final video.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with path to saved video
        """
        if not self.script_data:
            raise ValueError("No script generated. Call generate() first.")
        
        base_dir = output_path or 'outputs'
        file_handler = FileHandler(base_dir)
        unique_id = uuid.uuid4().hex
        
        # Create temporary directory for assets (using UUID)
        temp_dir = os.path.join(base_dir, 'video', f'temp_{unique_id}')
        os.makedirs(temp_dir, exist_ok=True)
        
        clips = []
        
        try:
            logger.info(f"Rendering {len(self.script_data)} scenes...")
            
            for i, scene in enumerate(self.script_data):
                # 1. Generate Slide Image
                slide_filename = f"slide_{i:03d}.png"
                slide_path = os.path.join(temp_dir, slide_filename)
                
                self.slide_gen.generate_slide(
                    title=scene.get('title', 'Untitled'),
                    content=scene.get('content', []),
                    output_path=slide_path
                )
                
                # 2. Generate Audio
                audio_filename = f"audio_{i:03d}.mp3"
                audio_path = os.path.join(temp_dir, audio_filename)
                
                narration = scene.get('narration', '')
                self.tts.generate_audio(narration, audio_path)
                
                # 3. Create Video Clip for this scene
                # Load audio to get duration
                try:
                    audio_clip = AudioFileClip(audio_path)
                    duration = audio_clip.duration
                    
                    # Create Image Clip with correct duration
                    video_clip = ImageClip(slide_path).with_duration(duration)
                    video_clip = video_clip.with_audio(audio_clip)
                    video_clip.fps = 24
                    
                    clips.append(video_clip)
                except Exception as e:
                    logger.error(f"Error processing scene {i}: {e}")
            
            if not clips:
                raise ValueError("No clips were successfully generated.")
            
            # 4. Concatenate and Save Final Video
            logger.info("Concatenating scenes into final video...")
            final_video = concatenate_videoclips(clips, method="compose")
            
            output_video_filename = f"presentation_{unique_id}.mp4"
            output_video_path = os.path.join(base_dir, 'video', output_video_filename)
            
            # Write video file
            final_video.write_videofile(
                output_video_path, 
                fps=24, 
                codec='libx264', 
                audio_codec='aac',
                logger=None # Suppress moviepy detailed logs
            )
            
            logger.info(f"Video saved to: {output_video_path}")
            return {'video_path': output_video_path}
            
        except Exception as e:
            logger.error(f"Video generation failed: {e}")
            # If generation failed, we might want to keep temp files for debugging
            # But normally we clean up to avoid clutter. 
            # Let's keep it clean: always cleanup unless configured otherwise.
            return {'error': str(e)}
        finally:
            # Cleanup resources
            try:
                # Close clips to release file handles
                for clip in clips:
                    try:
                        clip.close()
                        if clip.audio: clip.audio.close()
                    except: pass
                
                # Remove temporary directory
                if os.path.exists(temp_dir):
                    shutil.rmtree(temp_dir)
                    logger.debug(f"Cleaned up temporary directory: {temp_dir}")
            except Exception as e:
                logger.warning(f"Failed to cleanup temporary files: {e}")
