"""
Report generator - creates custom HTML research reports.
"""
import json
import uuid
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.report_prompt import REPORT_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)


class ReportGenerator(BaseGenerator):
    """Generate custom HTML research reports from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.html_content = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate report HTML using LLM.
        
        Returns:
            Dictionary with report and metadata
        """
        logger.info("Generating custom HTML research report...")
        
        # Generate report HTML using LLM
        response = self.generate_with_prompt(REPORT_PROMPT)
        
        # Try to extract JSON
        try:
            json_start = response.find('{')
            json_end = response.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                json_str = response[json_start:json_end]
                data = json.loads(json_str)
                
                # Extract HTML content
                self.html_content = data.get('html_content', '')
                
                if not self.html_content:
                    logger.warning("No html_content in JSON response")
                    self.html_content = None
                else:
                    logger.info("Successfully extracted HTML report from LLM response")
            else:
                logger.warning("Could not find JSON in response")
                self.html_content = None
            
        except json.JSONDecodeError as e:
            logger.error(f"JSON parsing error: {e}")
            logger.warning("Storing raw response as fallback")
            self.html_content = None
        
        return {
            'type': 'report',
            'has_html': self.html_content is not None,
            'generated_at': datetime.now().isoformat()
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save report as standalone HTML file.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with path to saved HTML file
        """
        if not self.html_content:
            raise ValueError("No report generated. Call generate() first or LLM failed to generate valid HTML.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        # Save the HTML file directly
        html_filename = f"report_{unique_id}.html"
        html_path = file_handler.save_text(
            self.html_content,
            'reports',
            html_filename
        )
        result['html_path'] = html_path
        logger.info(f"Report HTML saved to: {html_path}")
            
        return result

