"""
Quiz generator - creates assessment questions.
"""
import json
import uuid
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.quiz_prompt import QUIZ_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)


class QuizGenerator(BaseGenerator):
    """Generate quiz questions from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.quiz = None
        self.quiz_json = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate quiz using LLM.
        
        Returns:
            Dictionary with quiz and metadata
        """
        print("Generating quiz questions...")
        
        # Generate quiz using LLM
        response = self.generate_with_prompt(QUIZ_PROMPT)
        
        # Try to extract JSON
        try:
            json_start = response.find('{')
            json_end = response.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                json_str = response[json_start:json_end]
                self.quiz_json = json.loads(json_str)
                self.quiz = self.quiz_json.get('quiz', {})
            else:
                self.quiz = response
            
        except json.JSONDecodeError:
            logger.warning("Could not parse JSON, storing raw response")
            self.quiz = response
        
        return {
            'type': 'quiz',
            'quiz': self.quiz,
            'generated_at': datetime.now().isoformat(),
            'question_count': self.quiz.get('total_questions', 0) if isinstance(self.quiz, dict) else 0
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save quiz to file.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with paths to saved files
        """
        if not self.quiz:
            raise ValueError("No quiz generated. Call generate() first.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        if self.quiz_json:
            # Generate interactive HTML
            html_content = self._generate_quiz_html(self.quiz_json)
            html_filename = f"quiz_{unique_id}.html"
            html_path = file_handler.save_text(
                html_content,
                'quizzes',
                html_filename
            )
            result['html_path'] = html_path
            logger.info(f"Quiz interactive HTML saved to: {html_path}")
            
        return result
    
    def _generate_quiz_html(self, quiz_data: Dict[str, Any]) -> str:
        """Generate standalone HTML for the quiz."""
        import json
        quiz_json_str = json.dumps(quiz_data)
        
        return f"""<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{quiz_data.get('quiz', {}).get('title', 'Research Quiz')}</title>
    <style>
        :root {{
            --primary-color: #4F46E5; /* Indigo-600 */
            --bg-color: #F9FAFB;
            --card-bg: #FFFFFF;
            --text-main: #111827;
            --text-secondary: #6B7280;
            --option-bg: #F3F4F6;
            --option-hover: #E5E7EB;
            --correct-bg: #D1FAE5;
            --correct-border: #10B981;
            --wrong-bg: #FEE2E2;
            --wrong-border: #EF4444;
        }}
        body {{
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background-color: var(--bg-color);
            color: var(--text-main);
            margin: 0;
            padding: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
        }}
        .container {{
            width: 100%;
            max-width: 800px;
            background: var(--card-bg);
            padding: 40px;
            border-radius: 16px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05), 0 2px 4px -1px rgba(0, 0, 0, 0.03);
            position: relative;
            min-height: 600px;
        }}
        .header {{
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }}
        .title {{
            font-size: 1.5rem;
            font-weight: 700;
        }}
        .progress {{
            font-size: 0.9rem;
            color: var(--text-secondary);
        }}
        .question-card {{
            margin-bottom: 30px;
        }}
        .question-text {{
            font-size: 1.25rem;
            font-weight: 600;
            line-height: 1.6;
            margin-bottom: 24px;
        }}
        .options {{
            display: flex;
            flex-direction: column;
            gap: 12px;
        }}
        .option {{
            padding: 16px 20px;
            background-color: var(--option-bg);
            border: 2px solid transparent;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.2s ease;
            font-size: 1rem;
            display: flex;
            align-items: center;
        }}
        .option:hover:not(.disabled) {{
            background-color: var(--option-hover);
        }}
        .option.selected {{
            border-color: var(--primary-color);
            background-color: #EEF2FF;
        }}
        .option.correct {{
            background-color: var(--correct-bg);
            border-color: var(--correct-border);
            color: #065F46;
        }}
        .option.wrong {{
            background-color: var(--wrong-bg);
            border-color: var(--wrong-border);
            color: #991B1B;
        }}
        .option-label {{
            font-weight: 600;
            margin-right: 12px;
            min-width: 24px;
        }}
        .footer {{
            display: flex;
            justify-content: space-between;
            margin-top: 40px;
            align-items: center;
        }}
        .btn {{
            padding: 10px 24px;
            border-radius: 9999px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            border: none;
            outline: none;
        }}
        .btn-primary {{
            background-color: var(--primary-color);
            color: white;
            box-shadow: 0 4px 6px rgba(79, 70, 229, 0.2);
        }}
        .btn-primary:hover {{
            background-color: #4338CA;
        }}
        .btn-outline {{
            background-color: transparent;
            border: 1px solid #D1D5DB;
            color: var(--text-secondary);
        }}
        .btn-outline:hover {{
            border-color: var(--text-main);
            color: var(--text-main);
        }}
        .hint-box {{
            margin-top: 20px;
            padding: 16px;
            background-color: #FEF3C7;
            border-radius: 8px;
            color: #92400E;
            font-size: 0.95rem;
            display: none;
            animation: fadeIn 0.3s;
        }}
        @keyframes fadeIn {{
            from {{ opacity: 0; transform: translateY(-5px); }}
            to {{ opacity: 1; transform: translateY(0); }}
        }}
        .summary {{
            text-align: center;
            display: none;
        }}
        .score {{
            font-size: 3rem;
            font-weight: 800;
            color: var(--primary-color);
            margin: 20px 0;
        }}
        .close-btn {{
            position: absolute;
            top: 20px;
            right: 20px;
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.5rem;
            color: var(--text-secondary);
        }}
    </style>
</head>
<body>
    <div class="container">
        <button class="close-btn" onclick="window.close()">×</button>
        
        <div id="quiz-interface">
            <div class="header">
                <div class="title">Research Quiz</div>
                <div class="progress" id="progress">1 / 10</div>
            </div>

            <div class="question-card">
                <div class="question-text" id="question-text">Loading...</div>
                <div class="options" id="options-container">
                    <!-- Options will be injected here -->
                </div>
                
                <div id="hint-display" class="hint-box"></div>
            </div>

            <div class="footer">
                <button class="btn btn-outline" id="hint-btn">Hint</button>
                <button class="btn btn-primary" id="next-btn">Next</button>
            </div>
        </div>

        <div id="result-interface" class="summary">
            <h2>Quiz Completed!</h2>
            <div class="score" id="final-score">0/0</div>
            <p>Great job reviewing the paper.</p>
            <button class="btn btn-primary" onclick="location.reload()">Restart Quiz</button>
        </div>
    </div>

    <script>
        const quizData = {quiz_json_str};
        const questions = quizData.quiz.questions;
        
        let currentIdx = 0;
        let score = 0;
        let selectedOption = null;
        let isAnswered = false;

        const ui = {{
            questionText: document.getElementById('question-text'),
            optionsContainer: document.getElementById('options-container'),
            progress: document.getElementById('progress'),
            hintBtn: document.getElementById('hint-btn'),
            hintDisplay: document.getElementById('hint-display'),
            nextBtn: document.getElementById('next-btn'),
            quizInterface: document.getElementById('quiz-interface'),
            resultInterface: document.getElementById('result-interface'),
            finalScore: document.getElementById('final-score')
        }};

        function initQuiz() {{
            currentIdx = 0;
            score = 0;
            showQuestion(0);
            
            ui.nextBtn.addEventListener('click', handleNext);
            ui.hintBtn.addEventListener('click', toggleHint);
        }}

        function showQuestion(index) {{
            const q = questions[index];
            isAnswered = false;
            selectedOption = null;
            ui.hintDisplay.style.display = 'none';
            ui.nextBtn.textContent = (index === questions.length - 1) ? 'Finish' : 'Next';
            ui.nextBtn.disabled = true;
            ui.nextBtn.style.opacity = '0.5';
            
            ui.progress.textContent = `${{index + 1}} / ${{questions.length}}`;
            ui.questionText.textContent = q.question;
            
            ui.optionsContainer.innerHTML = '';
            
            if (q.type === 'multiple_choice' || q.type === 'true_false') {{
                const opts = q.options || ['True', 'False'];
                opts.forEach((opt, i) => {{
                    const el = document.createElement('div');
                    el.className = 'option';
                    const letter = String.fromCharCode(65 + i);
                    
                    // Handle True/False logic if needed, but assuming options array provided
                    // For pure T/F without options array, we synthesize
                    const text = opt;
                    
                    el.innerHTML = `<span class="option-label">${{letter}}.</span> ${{text}}`;
                    el.onclick = () => selectOption(i, letter, q);
                    ui.optionsContainer.appendChild(el);
                }});
            }} else {{
                ui.optionsContainer.innerHTML = '<div style="padding:20px; color:#666;">Short answer questions are for self-reflection.</div>';
                ui.nextBtn.disabled = false;
                ui.nextBtn.style.opacity = '1';
                isAnswered = true; 
            }}
        }}

        function selectOption(index, letter, question) {{
            if (isAnswered) return;
            
            isAnswered = true;
            const options = document.querySelectorAll('.option');
            const correctLetter = question.correct_answer === true ? 'A' : (question.correct_answer === false ? 'B' : question.correct_answer); 
            // Normalize correct answer. Some JSON might correct_answer: "A".
            // If True/False, usually True=A, False=B in our rendered list.
            
            let isCorrect = (String(letter) === String(correctLetter));
            
            // Highlight selected
            options[index].classList.add(isCorrect ? 'correct' : 'wrong');
            
            // If wrong, highlight correct
            if (!isCorrect) {{
                const correctIdx = correctLetter.charCodeAt(0) - 65;
                if (options[correctIdx]) options[correctIdx].classList.add('correct');
            }} else {{
                score++;
            }}
            
            // Disable all
            options.forEach(opt => opt.classList.add('disabled'));
            
            // Enable next
            ui.nextBtn.disabled = false;
            ui.nextBtn.style.opacity = '1';
        }}

        function handleNext() {{
            if (currentIdx < questions.length - 1) {{
                currentIdx++;
                showQuestion(currentIdx);
            }} else {{
                showResult();
            }}
        }}

        function showResult() {{
            ui.quizInterface.style.display = 'none';
            ui.resultInterface.style.display = 'block';
            ui.finalScore.textContent = `${{score}} / ${{questions.length}}`;
        }}

        function toggleHint() {{
            const q = questions[currentIdx];
            if (ui.hintDisplay.style.display === 'block') {{
                ui.hintDisplay.style.display = 'none';
            }} else {{
                ui.hintDisplay.textContent = q.hint || q.explanation || "No hint available.";
                ui.hintDisplay.style.display = 'block';
            }}
        }}

        initQuiz();
    </script>
</body>
</html>
"""
