"""
Mental Map generator - creates mind maps from academic papers.
"""
import os
import uuid
import re
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.mental_map_prompt import MENTAL_MAP_PROMPT
from utils.file_handler import FileHandler

from utils.logger import get_logger

logger = get_logger(__name__)


class MentalMapGenerator(BaseGenerator):
    """Generate mental/mind map from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.mindmap_text = None
        self.markdown_code = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate mind map structure using LLM.
        
        Returns:
            Dictionary with mind map text and metadata
        """
        logger.info("Generating mental map...")
        
        # Generate mind map using LLM
        full_response = self.generate_with_prompt(MENTAL_MAP_PROMPT)
        
        # Try to extract Markdown code if present
        # Look for ```markdown ... ``` blocks
        md_pattern = r'```markdown\n(.*?)\n```'
        md_match = re.search(md_pattern, full_response, re.DOTALL)
        
        if md_match:
            self.markdown_code = md_match.group(1)
        else:
            # If no code block, assume the whole text (or parts of it) might be structured
            # But usually LLM follows instructions. Fallback: try finding first #
            if '#' in full_response:
                self.markdown_code = full_response[full_response.find('#'):]
        
        self.mindmap_text = full_response
        
        return {
            'type': 'mental_map',
            'mindmap': self.mindmap_text,
            'format': 'markmap' if self.markdown_code else 'text',
            'generated_at': datetime.now().isoformat()
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save mind map to file (Markdown and HTML).
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with paths to saved files
        """
        if not self.mindmap_text:
            raise ValueError("No mind map generated. Call generate() first.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        if self.markdown_code:
            # Skip saving intermediate .md file as requested
            # Use self.markdown_code directly for HTML generation
            
            # Generate HTML (Markmap)
            html_content = f"""<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Research Mind Map</title>
<style>
body, html {{ margin: 0; padding: 0; width: 100%; height: 100%; overflow: hidden; }}
.markmap {{ width: 100%; height: 100vh; }}
</style>
<script src="https://cdn.jsdelivr.net/npm/markmap-autoloader"></script>
</head>
<body>
<div class="markmap">
{self.markdown_code}
</div>
</body>
</html>
"""
            html_filename = f"mindmap_{unique_id}.html"
            html_path = file_handler.save_text(
                html_content,
                'mental_maps',
                html_filename
            )
            logger.info(f"Mind map interactive HTML saved to: {html_path}")
            result['html_path'] = html_path
            
        else:
            # Fallback: Save full text response
            txt_filename = f"mindmap_{unique_id}.txt"
            txt_path = file_handler.save_text(
                self.mindmap_text,
                'mental_maps',
                txt_filename
            )
            logger.info(f"Mind map text saved to: {txt_path}")
            result['text_path'] = txt_path
        
        return result
