"""
Infographic generator - creates visual infographics with AI-generated HTML+CSS.
"""
import json
import uuid
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.infographic_prompt import INFOGRAPHIC_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)


class InfographicGenerator(BaseGenerator):
    """Generate HTML infographics from academic paper using LLM."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.html_content = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate complete HTML infographic using LLM.
        
        Returns:
            Dictionary with infographic metadata
        """
        logger.info("Generating HTML infographic with AI-designed layout...")
        
        # Generate infographic using LLM
        response = self.generate_with_prompt(INFOGRAPHIC_PROMPT)
        
        # Try to extract JSON containing html_content
        try:
            json_start = response.find('{')
            json_end = response.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                json_str = response[json_start:json_end]
                data = json.loads(json_str)
                self.html_content = data.get('html_content', '')
                
                if not self.html_content:
                    logger.error("LLM response missing 'html_content' field")
                    raise ValueError("Invalid LLM response: missing html_content")
                    
                logger.info("Successfully extracted HTML content from LLM response")
            else:
                logger.error("Could not find JSON in LLM response")
                raise ValueError("LLM response does not contain valid JSON")
            
        except json.JSONDecodeError as e:
            logger.error(f"Failed to parse JSON from LLM response: {e}")
            raise ValueError(f"Invalid JSON in LLM response: {e}")
        
        return {
            'type': 'infographic',
            'generated_at': datetime.now().isoformat(),
            'has_html': bool(self.html_content)
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save infographic as standalone HTML file.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with path to saved HTML file
        """
        if not self.html_content:
            raise ValueError("No infographic generated. Call generate() first or LLM failed to generate valid HTML.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        # Save the HTML file directly
        html_filename = f"infographic_{unique_id}.html"
        html_path = file_handler.save_text(
            self.html_content,
            'infographics',
            html_filename
        )
        result['html_path'] = html_path
        logger.info(f"Infographic HTML saved to: {html_path}")
            
        return result

