"""
Flashcard generator - creates interactive study flashcards.
"""
import json
import uuid
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.flashcard_prompt import FLASHCARD_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)


class FlashcardGenerator(BaseGenerator):
    """Generate interactive study flashcards from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.flashcards = None
        self.flashcards_json = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate flashcards using LLM.
        
        Returns:
            Dictionary with flashcards and metadata
        """
        print("Generating flashcards...")
        
        # Generate flashcards using LLM
        response = self.generate_with_prompt(FLASHCARD_PROMPT)
        
        # Try to extract JSON
        try:
            json_start = response.find('{')
            json_end = response.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                json_str = response[json_start:json_end]
                self.flashcards_json = json.loads(json_str)
                self.flashcards = self.flashcards_json.get('flashcards', {})
            else:
                self.flashcards = response
            
        except json.JSONDecodeError:
            logger.warning("Could not parse JSON, storing raw response")
            self.flashcards = response
        
        card_count = 0
        if isinstance(self.flashcards, dict):
            card_count = len(self.flashcards.get('cards', []))
        
        return {
            'type': 'flashcard',
            'flashcards': self.flashcards,
            'count': card_count,
            'generated_at': datetime.now().isoformat()
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save flashcards as interactive HTML.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with path to saved HTML file
        """
        if not self.flashcards:
            raise ValueError("No flashcards generated. Call generate() first.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        if self.flashcards_json:
            # Generate interactive HTML
            html_content = self._generate_flashcard_html(self.flashcards_json)
            html_filename = f"flashcards_{unique_id}.html"
            html_path = file_handler.save_text(
                html_content,
                'flashcards',
                html_filename
            )
            result['html_path'] = html_path
            logger.info(f"Flashcards HTML saved to: {html_path}")
            
        return result
    
    def _generate_flashcard_html(self, data: Dict[str, Any]) -> str:
        """Generate standalone HTML for interactive flashcards."""
        import json
        flashcards_data = data.get('flashcards', {})
        title = flashcards_data.get('title', 'Research Flashcards')
        cards = flashcards_data.get('cards', [])
        
        return f"""<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{title}</title>
    <style>
        * {{
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }}
        
        body {{
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #c7b3ff 0%, #b3ffd9 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }}
        
        .container {{
            width: 100%;
            max-width: 600px;
        }}
        
        .header {{
            text-align: center;
            margin-bottom: 40px;
        }}
        
        .title {{
            font-size: 1.5rem;
            font-weight: 700;
            color: #1f2937;
            margin-bottom: 10px;
        }}
        
        .instructions {{
            font-size: 0.9rem;
            color: #6b7280;
        }}
        
        .card-container {{
            perspective: 1000px;
            margin-bottom: 40px;
        }}
        
        .card {{
            width: 100%;
            height: 400px;
            position: relative;
            transform-style: preserve-3d;
            transition: transform 0.6s;
            cursor: pointer;
        }}
        
        .card.flipped {{
            transform: rotateY(180deg);
        }}
        
        .card-face {{
            position: absolute;
            width: 100%;
            height: 100%;
            backface-visibility: hidden;
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
        }}
        
        .card-front {{
            background: #2b2b2b;
            color: white;
        }}
        
        .card-back {{
            background: white;
            color: #1f2937;
            transform: rotateY(180deg);
        }}
        
        .card-content {{
            text-align: center;
            font-size: 1.3rem;
            line-height: 1.6;
        }}
        
        .card-back .card-content {{
            font-size: 1.1rem;
        }}
        
        .flip-hint {{
            position: absolute;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            font-size: 0.85rem;
            opacity: 0.7;
        }}
        
        .controls {{
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 40px;
        }}
        
        .nav-btn {{
            width: 50px;
            height: 50px;
            border-radius: 50%;
            border: none;
            background: white;
            color: #4f46e5;
            font-size: 1.5rem;
            cursor: pointer;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
        }}
        
        .nav-btn:hover:not(:disabled) {{
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.15);
            transform: translateY(-2px);
        }}
        
        .nav-btn:disabled {{
            opacity: 0.3;
            cursor: not-allowed;
        }}
        
        .progress {{
            text-align: center;
            font-size: 0.9rem;
            color: #374151;
        }}
        
        .progress-bar {{
            width: 100%;
            height: 4px;
            background: rgba(255, 255, 255, 0.3);
            border-radius: 2px;
            margin-top: 30px;
            overflow: hidden;
        }}
        
        .progress-fill {{
            height: 100%;
            background: #4f46e5;
            transition: width 0.3s;
        }}
        
        .shortcuts {{
            text-align: center;
            margin-top: 20px;
            font-size: 0.8rem;
            color: #6b7280;
        }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 class="title">{title}</h1>
            <p class="instructions">Click card to flip • Use arrows to navigate</p>
        </div>
        
        <div class="card-container">
            <div class="card" id="flashcard" onclick="flipCard()">
                <div class="card-face card-front">
                    <div class="card-content" id="question">Loading...</div>
                    <div class="flip-hint">Click to reveal answer</div>
                </div>
                <div class="card-face card-back">
                    <div class="card-content" id="answer">Loading...</div>
                    <div class="flip-hint">Click to see question</div>
                </div>
            </div>
        </div>
        
        <div class="controls">
            <button class="nav-btn" id="prev-btn" onclick="previousCard()">←</button>
            <div class="progress" id="progress">1 / {len(cards)}</div>
            <button class="nav-btn" id="next-btn" onclick="nextCard()">→</button>
        </div>
        
        <div class="progress-bar">
            <div class="progress-fill" id="progress-fill"></div>
        </div>
        
        <div class="shortcuts">
            Keyboard: ← / → to navigate • Space to flip
        </div>
    </div>

    <script>
        const cards = {json.dumps([{'question': c.get('question', ''), 'answer': c.get('answer', '')} for c in cards])};
        
        let currentIndex = 0;
        let isFlipped = false;

        function showCard(index) {{
            const card = cards[index];
            document.getElementById('question').textContent = card.question;
            document.getElementById('answer').textContent = card.answer;
            document.getElementById('progress').textContent = `${{index + 1}} / ${{cards.length}}`;
            
            // Update progress bar
            const progressPercent = ((index + 1) / cards.length) * 100;
            document.getElementById('progress-fill').style.width = `${{progressPercent}}%`;
            
            // Update button states
            document.getElementById('prev-btn').disabled = index === 0;
            document.getElementById('next-btn').disabled = index === cards.length - 1;
            
            // Reset flip state
            if (isFlipped) {{
                flipCard();
            }}
        }}

        function flipCard() {{
            const cardElement = document.getElementById('flashcard');
            cardElement.classList.toggle('flipped');
            isFlipped = !isFlipped;
        }}

        function nextCard() {{
            if (currentIndex < cards.length - 1) {{
                currentIndex++;
                showCard(currentIndex);
            }}
        }}

        function previousCard() {{
            if (currentIndex > 0) {{
                currentIndex--;
                showCard(currentIndex);
            }}
        }}

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {{
            if (e.key === 'ArrowRight') nextCard();
            if (e.key === 'ArrowLeft') previousCard();
            if (e.key === ' ') {{
                e.preventDefault();
                flipCard();
            }}
        }});

        // Initialize
        showCard(0);
    </script>
</body>
</html>
"""

