"""
Data Table generator - creates formatted HTML tables from research papers.
"""
import json
import uuid
from typing import Dict, Any
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.datatable_prompt import DATATABLE_PROMPT
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)


class DatatableGenerator(BaseGenerator):
    """Generate HTML data tables from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.table = None
        self.table_json = None
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate data table using LLM.
        
        Returns:
            Dictionary with table and metadata
        """
        print("Generating data table...")
        
        # Generate table using LLM
        response = self.generate_with_prompt(DATATABLE_PROMPT)
        
        # Try to extract JSON
        try:
            json_start = response.find('{')
            json_end = response.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                json_str = response[json_start:json_end]
                self.table_json = json.loads(json_str)
                self.table = self.table_json.get('datatable', {})
            else:
                self.table = response
            
        except json.JSONDecodeError:
            logger.warning("Could not parse JSON, storing raw response")
            self.table = response
        
        return {
            'type': 'datatable',
            'table': self.table,
            'generated_at': datetime.now().isoformat()
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        """
        Save data table as HTML.
        
        Args:
            output_path: Base output directory
            
        Returns:
            Dictionary with path to saved HTML file
        """
        if not self.table:
            raise ValueError("No table generated. Call generate() first.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        result = {}
        
        if self.table_json:
            # Generate HTML table
            html_content = self._generate_table_html(self.table_json)
            html_filename = f"datatable_{unique_id}.html"
            html_path = file_handler.save_text(
                html_content,
                'datatables',
                html_filename
            )
            result['html_path'] = html_path
            logger.info(f"Data table HTML saved to: {html_path}")
            
        return result
    
    def _generate_table_html(self, data: Dict[str, Any]) -> str:
        """Generate standalone HTML for the data table."""
        table_data = data.get('datatable', {})
        title = table_data.get('title', 'Research Data Table')
        subtitle = table_data.get('subtitle', '')
        description = table_data.get('description', '')
        headers = table_data.get('headers', [])
        rows = table_data.get('rows', [])
        footer = table_data.get('footer', '')
        
        # Build table headers
        headers_html = ''
        for header in headers:
            headers_html += f'<th>{header}</th>'
        
        # Build table rows
        rows_html = ''
        for row in rows:
            rows_html += '<tr>'
            for cell in row:
                rows_html += f'<td>{cell}</td>'
            rows_html += '</tr>'
        
        return f"""<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{title}</title>
    <style>
        * {{
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }}
        
        body {{
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #e9ecef 100%);
            padding: 40px 20px;
            min-height: 100vh;
        }}
        
        .container {{
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.08);
            overflow: hidden;
        }}
        
        .header {{
            padding: 40px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }}
        
        .title {{
            font-size: 1.8rem;
            font-weight: 700;
            margin-bottom: 10px;
            line-height: 1.3;
        }}
        
        .subtitle {{
            font-size: 0.95rem;
            opacity: 0.9;
            margin-bottom: 8px;
        }}
        
        .description {{
            font-size: 0.9rem;
            opacity: 0.85;
            font-style: italic;
        }}
        
        .table-wrapper {{
            padding: 40px;
            overflow-x: auto;
        }}
        
        table {{
            width: 100%;
            border-collapse: collapse;
            font-size: 0.9rem;
        }}
        
        thead {{
            background: #f8f9fa;
            position: sticky;
            top: 0;
            z-index: 10;
        }}
        
        th {{
            padding: 16px 12px;
            text-align: left;
            font-weight: 600;
            color: #374151;
            border-bottom: 2px solid #667eea;
            white-space: nowrap;
        }}
        
        td {{
            padding: 12px;
            border-bottom: 1px solid #e5e7eb;
            color: #1f2937;
        }}
        
        tbody tr:hover {{
            background: #f9fafb;
        }}
        
        tbody tr:nth-child(even) {{
            background: #fcfcfd;
        }}
        
        tbody tr:nth-child(even):hover {{
            background: #f3f4f6;
        }}
        
        .footer {{
            padding: 20px 40px;
            background: #f8f9fa;
            font-size: 0.85rem;
            color: #6b7280;
            border-top: 1px solid #e5e7eb;
        }}
        
        /* Responsive */
        @media (max-width: 768px) {{
            .title {{
                font-size: 1.3rem;
            }}
            
            .table-wrapper {{
                padding: 20px;
            }}
            
            th, td {{
                padding: 10px 8px;
                font-size: 0.85rem;
            }}
        }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 class="title">{title}</h1>
            {f'<p class="subtitle">{subtitle}</p>' if subtitle else ''}
            {f'<p class="description">{description}</p>' if description else ''}
        </div>
        
        <div class="table-wrapper">
            <table>
                <thead>
                    <tr>
                        {headers_html}
                    </tr>
                </thead>
                <tbody>
                    {rows_html}
                </tbody>
            </table>
        </div>
        
        {f'<div class="footer">{footer}</div>' if footer else ''}
    </div>
</body>
</html>
"""

