"""
Base generator class that all specific generators inherit from.
"""
import os
from abc import ABC, abstractmethod
from typing import Dict, Any
from dotenv import load_dotenv

# Updated imports for langchain 0.2+
from langchain_community.chat_models import ChatOpenAI
from langchain_google_genai import ChatGoogleGenerativeAI
from langchain_anthropic import ChatAnthropic
from langchain_core.messages import HumanMessage, SystemMessage

# Load environment variables
load_dotenv()


class BaseGenerator(ABC):
    """Base class for all resource generators."""
    
    def __init__(self, paper_content: str):
        """
        Initialize generator with paper content.
        
        Args:
            paper_content: Extracted text from the PDF paper
        """
        self.paper_content = paper_content
        self.llm = self._initialize_llm()
    
    def _initialize_llm(self):
        """
        Initialize the LLM based on environment configuration.
        
        Returns:
            Configured LLM instance
        """
        # Load configuration from environment variables
        provider = os.getenv('LLM_PROVIDER', 'google').lower()
        model_name = os.getenv('MODEL_NAME', 'gemini-pro')
        temperature = float(os.getenv('TEMPERATURE', '0.7'))
        max_tokens = int(os.getenv('MAX_TOKENS', '8000'))
        
        if provider == 'google':
            return ChatGoogleGenerativeAI(
                model=model_name,
                google_api_key=os.getenv('GOOGLE_API_KEY', ''),
                temperature=temperature,
                max_output_tokens=max_tokens
            )
        elif provider == 'openai':
            return ChatOpenAI(
                model=model_name,
                openai_api_key=os.getenv('OPENAI_API_KEY', ''),
                temperature=temperature,
                max_tokens=max_tokens
            )
        elif provider == 'anthropic':
            return ChatAnthropic(
                model=model_name,
                anthropic_api_key=os.getenv('ANTHROPIC_API_KEY', ''),
                temperature=temperature,
                max_tokens=max_tokens
            )
        else:
            raise ValueError(f"Unsupported LLM provider: {provider}")
    
    def generate_with_prompt(self, prompt_template: str, **kwargs) -> str:
        """
        Generate content using the LLM with a given prompt template.
        
        Args:
            prompt_template: The prompt template string
            **kwargs: Additional variables to format into the prompt
            
        Returns:
            Generated text from the LLM
        """
        # Format prompt with paper content and any additional kwargs
        formatted_prompt = prompt_template.format(
            paper_content=self.paper_content,
            **kwargs
        )
        
        # Generate response
        messages = [HumanMessage(content=formatted_prompt)]
        response = self.llm.invoke(messages)
        
        return response.content
    
    @abstractmethod
    def generate(self) -> Dict[str, Any]:
        """
        Generate the specific resource type.
        
        Must be implemented by subclasses.
        
        Returns:
            Dictionary containing the generated content and metadata
        """
        pass
    
    @abstractmethod
    def save(self, output_path: str) -> str:
        """
        Save the generated resource to file.
        
        Must be implemented by subclasses.
        
        Args:
            output_path: Directory to save the resource
            
        Returns:
            Path to saved file
        """
        pass
