import json
import asyncio
import os
import uuid
from typing import Dict, Any, List
from datetime import datetime
from .base_generator import BaseGenerator
from prompts.audio_prompt import AUDIO_PROMPT
from plugins.text_to_speech import TextToSpeech
from utils.file_handler import FileHandler
from utils.logger import get_logger

logger = get_logger(__name__)

class AudioGenerator(BaseGenerator):
    """Generate audio podcast from academic paper."""
    
    def __init__(self, paper_content: str):
        super().__init__(paper_content)
        self.script_data = None # List[Dict]
        self.tts = TextToSpeech()
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate podcast script using LLM.
        """
        logger.info("Generating audio podcast script...")
        
        # Generate script using LLM
        response = self.generate_with_prompt(AUDIO_PROMPT)
        
        # Clean response to ensure valid JSON
        cleaned_response = response.strip()
        if cleaned_response.startswith('```json'):
            cleaned_response = cleaned_response[7:]
        if cleaned_response.endswith('```'):
            cleaned_response = cleaned_response[:-3]
            
        try:
            self.script_data = json.loads(cleaned_response)
        except json.JSONDecodeError as e:
            logger.error(f"Failed to parse LLM JSON output: {e}")
            # Fallback for text wrapper
            try:
                start = cleaned_response.find('[')
                end = cleaned_response.rfind(']') + 1
                if start != -1 and end != 0:
                     json_str = cleaned_response[start:end]
                     # Remove trailing commas which are common LLM errors
                     import re
                     json_str = re.sub(r',(\s*])', r'\1', json_str)
                     self.script_data = json.loads(json_str)
                else:
                    raise ValueError("No JSON found")
            except Exception as e2:
                logger.error(f"Fallback JSON parsing failed: {e2}")
                logger.error(f"Full problematic response: {response}")
                raise ValueError(f"LLM did not return valid JSON for audio script. Raw (snippet): {cleaned_response[:100]}")

        return {
            'type': 'audio',
            'script_data': self.script_data,
            'generated_at': datetime.now().isoformat(),
            'turn_count': len(self.script_data)
        }
    
    def save(self, output_path: str = None) -> Dict[str, str]:
        if not self.script_data:
            raise ValueError("No script generated. Call generate() first.")
        
        file_handler = FileHandler(output_path or 'outputs')
        unique_id = uuid.uuid4().hex
        
        # Save script JSON
        script_filename = f"podcast_script_{unique_id}.json"
        script_path = file_handler.save_text(
            json.dumps(self.script_data, indent=2), 
            'audio', 
            script_filename
        )
        
        # Convert to audio
        logger.info("Converting to audio using Edge TTS...")
        audio_filename = f"podcast_{unique_id}.mp3"
        audio_path = file_handler.get_resource_path('audio', audio_filename)
        
        try:
            # We need to run async function from sync context
            asyncio.run(self.tts.generate_podcast_audio(self.script_data, audio_path))
            logger.info(f"Audio saved to: {audio_path}")
            
            # Delete intermediate script file if audio generation was successful
            if os.path.exists(script_path):
                os.remove(script_path)
                logger.debug(f"Removed intermediate script file: {script_path}")
                script_path = None # Do not return path to deleted file
                
        except Exception as e:
            logger.error(f"Could not generate audio file: {e}")
            audio_path = None
        
        return {
            'script_path': script_path, # Will be None if deleted
            'audio_path': audio_path
        }
