"""
ContentAutoGenerator - Core class for generating educational resources.

This module contains the main orchestrator class that can be used
independently in various contexts (CLI, Web API, Celery tasks, etc.).
"""
import os
from typing import List, Optional, Dict, Any
from dotenv import load_dotenv

from utils.logger import setup_logger, get_logger
from utils.pdf_parser import PDFParser
from utils.file_handler import FileHandler

from generators.audio_generator import AudioGenerator
from generators.video_generator import VideoGenerator
from generators.mental_map_generator import MentalMapGenerator
from generators.report_generator import ReportGenerator
from generators.flashcard_generator import FlashcardGenerator
from generators.quiz_generator import QuizGenerator
from generators.infographic_generator import InfographicGenerator
from generators.presentation_generator import PresentationGenerator
from generators.datatable_generator import DatatableGenerator

logger = get_logger(__name__)

# Load environment variables from .env file
load_dotenv()


class ContentAutoGenerator:
    """
    Main orchestrator for generating educational resources from academic papers.
    
    This class is designed to be used both as a standalone tool and as a task
    for async processing systems like Celery.
    
    Example:
        # Direct usage
        generator = ContentAutoGenerator(
            pdf_path="paper.pdf",
            resource_types=['audio', 'report'],
            output_dir="./outputs",
            llm_provider="google"
        )
        generator.run()
        
        # Or step by step
        generator = ContentAutoGenerator(pdf_path="paper.pdf")
        generator.parse_pdf()
        generator.generate_all(resource_types=['audio'])
        results = generator.get_results()
    """
    
    # Generator registry
    GENERATOR_REGISTRY = {
        'audio': AudioGenerator,
        'video': VideoGenerator,
        'mental_map': MentalMapGenerator,
        'report': ReportGenerator,
        'flashcard': FlashcardGenerator,
        'quiz': QuizGenerator,
        'infographic': InfographicGenerator,
        'presentation': PresentationGenerator,
        'datatable': DatatableGenerator,
    }
    
    def __init__(
        self,
        pdf_path: str,
        resource_types: Optional[List[str]] = None,
        output_dir: Optional[str] = None,
        llm_provider: Optional[str] = None,
        model_name: Optional[str] = None,
        temperature: Optional[float] = None
    ):
        """
        Initialize content generator.
        
        Args:
            pdf_path: Path to the PDF academic paper
            resource_types: List of resource types to generate. 
                          If None, generates all enabled types from .env config.
            output_dir: Custom output directory. If None, uses OUTPUT_DIR from .env
            llm_provider: LLM provider ('google', 'openai', 'anthropic'). 
                         If None, uses LLM_PROVIDER from .env
            model_name: Specific model name. If None, uses MODEL_NAME from .env
            temperature: Model temperature. If None, uses TEMPERATURE from .env
        """
        # Setup logger (ensures logger is configured when class is instantiated)
        setup_logger(__name__)
        
        self.pdf_path = pdf_path
        self.resource_types = resource_types
        self.paper_content = None
        self.paper_info = None
        self.results = {}
        
        # Validate PDF exists
        if not os.path.exists(pdf_path):
            raise FileNotFoundError(f"PDF file not found: {pdf_path}")
        
        # Load configuration from environment variables or use defaults
        self.output_dir = output_dir or os.getenv('OUTPUT_DIR', './outputs')
        self.llm_provider = llm_provider or os.getenv('LLM_PROVIDER', 'google')
        self.model_name = model_name or os.getenv('MODEL_NAME', 'gemini-pro')
        self.temperature = temperature if temperature is not None else float(os.getenv('TEMPERATURE', '0.7'))
        
        # Ensure output directory exists
        self._ensure_output_dir()
        
        logger.info(f"ContentAutoGenerator initialized for: {pdf_path}")
        logger.info(f"Output directory: {self.output_dir}")
        logger.info(f"LLM: {self.llm_provider}/{self.model_name}")
    
    def _ensure_output_dir(self):
        """Ensure output directory exists."""
        os.makedirs(self.output_dir, exist_ok=True)
    
    def parse_pdf(self) -> Dict[str, Any]:
        """
        Parse the PDF and extract content.
        
        Returns:
            Dictionary containing paper info (title, page_count, text, etc.)
        """
        logger.info("="*70)
        logger.info(f"PARSING PDF: {self.pdf_path}")
        logger.info("="*70)
        
        parser = PDFParser(self.pdf_path)
        self.paper_info = parser.extract_all()
        self.paper_content = self.paper_info['text']
        
        logger.info("✓ PDF parsed successfully")
        logger.info(f"  - Title: {self.paper_info.get('title', 'N/A')}")
        logger.info(f"  - Pages: {self.paper_info.get('page_count', 'N/A')}")
        logger.info(f"  - Characters: {len(self.paper_content)}")
        
        return self.paper_info
    
    def generate_all(self, resource_types: Optional[List[str]] = None) -> Dict[str, Any]:
        """
        Generate all requested resource types.
        
        Args:
            resource_types: List of specific types to generate. 
                          If None, uses self.resource_types or all enabled types.
        
        Returns:
            Dictionary mapping resource types to their generation results
        """
        if not self.paper_content:
            raise ValueError("PDF not parsed. Call parse_pdf() first.")
        
        # Determine which types to generate
        types_to_generate = resource_types or self.resource_types
        
        if types_to_generate:
            # Generate specific types
            generators_to_run = {
                k: v for k, v in self.GENERATOR_REGISTRY.items() 
                if k in types_to_generate
            }
        else:
            # Generate all enabled types from config
            enabled_types = {
                'audio': os.getenv("ENABLE_AUDIO", "true").lower() == "true",
                'video': os.getenv("ENABLE_VIDEO", "true").lower() == "true",
                'mental_map': os.getenv("ENABLE_MENTAL_MAP", "true").lower() == "true",
                'report': os.getenv("ENABLE_REPORT", "true").lower() == "true",
                'flashcard': os.getenv("ENABLE_FLASHCARD", "true").lower() == "true",
                'quiz': os.getenv("ENABLE_QUIZ", "true").lower() == "true",
                'infographic': os.getenv("ENABLE_INFOGRAPHIC", "true").lower() == "true",
                'presentation': os.getenv("ENABLE_PRESENTATION", "true").lower() == "true",
                'datatable': os.getenv("ENABLE_DATATABLE", "true").lower() == "true",
            }
            generators_to_run = {
                k: v for k, v in self.GENERATOR_REGISTRY.items()
                if enabled_types.get(k, False)
            }
        
        # Generate each type
        for resource_type, GeneratorClass in generators_to_run.items():
            try:
                logger.info("-"*70)
                logger.info(f"GENERATING: {resource_type.upper().replace('_', ' ')}")
                logger.info("-"*70)
                
                # Initialize generator
                generator = GeneratorClass(self.paper_content)
                
                # Generate content
                result = generator.generate()
                
                # Save content
                save_result = generator.save(self.output_dir)
                
                # Store results
                self.results[resource_type] = {
                    'status': 'success',
                    'files': save_result,
                    'metadata': result
                }
                
                logger.info(f"✓ {resource_type} generated successfully")
                
            except Exception as e:
                logger.error(f"✗ Error generating {resource_type}: {e}")
                self.results[resource_type] = {
                    'status': 'error',
                    'error': str(e)
                }
        
        return self.results
    
    def get_results(self) -> Dict[str, Any]:
        """
        Get generation results.
        
        Returns:
            Dictionary mapping resource types to their results
        """
        return self.results
    
    def get_summary(self) -> Dict[str, Any]:
        """
        Get summary statistics of generation.
        
        Returns:
            Dictionary with summary information
        """
        successful = sum(1 for r in self.results.values() if r['status'] == 'success')
        failed = sum(1 for r in self.results.values() if r['status'] == 'error')
        total = len(self.results)
        
        return {
            'total': total,
            'successful': successful,
            'failed': failed,
            'success_rate': successful / total if total > 0 else 0,
            'results': self.results
        }
    
    def print_summary(self):
        """Print summary of all generated resources to logger."""
        logger.info("="*70)
        logger.info("GENERATION SUMMARY")
        logger.info("="*70)
        
        summary = self.get_summary()
        logger.info(f"Successfully generated: {summary['successful']}/{summary['total']} resource types")
        logger.info("")
        
        for resource_type, result in self.results.items():
            status_symbol = "✓" if result['status'] == 'success' else "✗"
            logger.info(f"{status_symbol} {resource_type.upper().replace('_', ' ')}")
            
            if result['status'] == 'success':
                for file_type, file_path in result['files'].items():
                    if file_path:
                        logger.info(f"    → {file_type}: {file_path}")
            else:
                logger.info(f"    → Error: {result.get('error', 'Unknown error')}")
        
        logger.info("="*70)
        logger.info(f"All outputs saved to: {self.output_dir}")
        logger.info("="*70)
    
    def run(self) -> Dict[str, Any]:
        """
        Run the complete generation pipeline: parse PDF → generate → return results.
        
        This is a convenience method for one-shot execution.
        
        Returns:
            Dictionary with generation results
        """
        try:
            # Parse PDF
            self.parse_pdf()
            
            # Generate resources
            self.generate_all()
            
            # Print summary
            self.print_summary()
            
            return self.get_summary()
            
        except Exception as e:
            logger.error(f"Fatal error in generation pipeline: {e}")
            raise
