import os
import sys
from urllib.parse import unquote
from config import Config

# Hardcoded match with upload.py and docker-compose volume
UPLOAD_ROOT = Config.UPLOAD_FOLDER

def delete_file(file_url):
    """
    Delete a file from the server given its URL.
    Args:
        file_url (str): The relative URL of the file (e.g. /use_cases/image.png)
    """
    if not file_url:
        return
    
    try:
        # Normalize the URL to get the filename/path
        clean_path = normalize_file_path(file_url)
        
        # If it looks like an external URL after normalization
        if clean_path.startswith('http') or clean_path.startswith('data:'):
            return
        
        # Candidate roots to check
        candidates = [UPLOAD_ROOT]
        if '/app/assets' not in candidates:
             candidates.append('/app/assets')
        if 'assets' not in candidates and os.path.exists('assets'):
             candidates.append('assets')
             
        deleted = False
        for root in candidates:
            if not root: continue
            file_path = os.path.join(root, clean_path)
            
            if os.path.exists(file_path):
                os.remove(file_path)
                print(f"DEBUG: Successfully deleted {file_path}", file=sys.stderr)
                deleted = True

                break
        
        if not deleted:
            print(f"DEBUG: File not found in candidates {candidates}: {clean_path}", file=sys.stderr)
            
    except Exception as e:
        print(f"ERROR: Failed to delete file {file_url}: {e}", file=sys.stderr)

def delete_files(file_list):
    """
    Delete multiple files from the server.
    Args:
        file_list (list): List of relative URLs/filenames.
    """
    if not file_list:
        return
        
    for f in file_list:
        if f:
            delete_file(f)

def normalize_file_path(file_url):
    """
    Convert a full URL or relative path to the storage filename/path.
    Strips the domain and /assets/ prefix.
    """
    if not file_url:
        return None
        
    # If it's a data URI, keep it (though usually not saved to DB for files)
    if file_url.startswith('data:'):
        return file_url
        
    # If it's an external URL (http/https) but NOT our own domain/assets, 
    # we conceivably should keep it as is (external link). 
    # BUT, the problem is detecting "our own domain" reliably if BASE_URL varies.
    # However, if it matches our /assets/ pattern, we assume it's ours.
    
    # Heuristic: if it contains '/assets/', take everything after the last '/assets/'
    if '/assets/' in file_url:
        return file_url.split('/assets/')[-1]
        
    # If it starts with http but no /assets/, it's likely purely external
    if file_url.startswith('http'):
        return file_url
        
    # Standard cleanup for relative paths
    clean = file_url.lstrip('/')
    if clean.startswith('assets/'):
        clean = clean.replace('assets/', '', 1)
        
    return clean

def get_asset_url(filename):
    """
    Construct the full URL for an asset.
    Args:
        filename (str): The filename or path stored in DB.
    Returns:
        str: The full URL to access the asset.
    """
    if not filename:
        return None
        
    if filename.startswith('http') or filename.startswith('https') or filename.startswith('data:'):
        return filename
        
    # Get BASE_URL from env, default to localhost:8000 (common dev setup)
    # This assumes the assets are served under /assets/
    base_url = os.environ.get('BASE_URL', 'http://localhost:8000')
    
    # Handle filename that might already start with / or /assets/ (legacy)
    # New convention is flat filename
    
    clean_name = filename.lstrip('/')
    if clean_name.startswith('assets/'):
         clean_name = clean_name.replace('assets/', '', 1)
         
    return f"{base_url.rstrip('/')}/assets/{clean_name}"
