"""
Use Case Attributes routes - CRUD operations for keyword, framework, model, hardware tables.
"""

from flask import Blueprint, request, jsonify
from db import get_db_connection

use_case_attributes_bp = Blueprint('use_case_attributes', __name__)

TABLES = {
    'keywords': 'usecase_schema.keyword',
    'frameworks': 'usecase_schema.framework',
    'models': 'usecase_schema.model',
    'hardwares': 'usecase_schema.hardware'
}

@use_case_attributes_bp.route('/use-case-attributes/<string:type>', methods=['GET'])
def get_attributes(type):
    """Get all attributes of a specific type."""
    if type not in TABLES:
        return jsonify({'error': 'Invalid attribute type'}), 400
    
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute(f"SELECT * FROM {TABLES[type]} ORDER BY name")
        items = cursor.fetchall()
        return jsonify(list(items)), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()

@use_case_attributes_bp.route('/use-case-attributes/<string:type>', methods=['POST'])
def create_attribute(type):
    """Create new attribute."""
    if type not in TABLES:
        return jsonify({'error': 'Invalid attribute type'}), 400
    
    data = request.get_json()
    if not data or 'name' not in data:
         return jsonify({'error': 'Name is required'}), 400

    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute(f"""
            INSERT INTO {TABLES[type]} (name)
            VALUES (%s)
            RETURNING *
        """, (data['name'],))
        item = cursor.fetchone()
        conn.commit()
        return jsonify(dict(item)), 201
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()

@use_case_attributes_bp.route('/use-case-attributes/<string:type>/<uuid:id>', methods=['PUT'])
def update_attribute(type, id):
    """Update attribute by ID."""
    if type not in TABLES:
        return jsonify({'error': 'Invalid attribute type'}), 400

    data = request.get_json()
    if not data or 'name' not in data:
         return jsonify({'error': 'Name is required'}), 400

    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute(f"""
            UPDATE {TABLES[type]} 
            SET name = %s
            WHERE id = %s
            RETURNING *
        """, (data['name'], str(id)))
        item = cursor.fetchone()
        conn.commit()
        if item:
            return jsonify(dict(item)), 200
        return jsonify({'error': 'Item not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()

@use_case_attributes_bp.route('/use-case-attributes/<string:type>/<uuid:id>', methods=['DELETE'])
def delete_attribute(type, id):
    """Delete attribute by ID."""
    if type not in TABLES:
        return jsonify({'error': 'Invalid attribute type'}), 400

    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute(f"DELETE FROM {TABLES[type]} WHERE id = %s RETURNING id", (str(id),))
        deleted = cursor.fetchone()
        conn.commit()
        if deleted:
            return jsonify({'message': 'Item deleted'}), 200
        return jsonify({'error': 'Item not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
